
#include "platform.h"
#include <string.h>
#include <math.h>
#include <iostream>
#include <sstream>
#include <fstream>

#include <zlib.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "svector.h"
#include "random.h"

using std::string;

#include "basics.h"

#include <OpenGL/gl.h>
#include <OpenGL/glu.h>
#include <CoreServices/CoreServices.h>

#include "endian.h"
#include "files.h"
#include "enabler.h"

enablerst enabler;

static AGLContext contextWindowed = NULL;
static AGLContext contextFullscreen = NULL;

static const GLint attrsWindowed[] = 
{
   AGL_RGBA,
   AGL_PIXEL_SIZE, 32,
   // AGL_DEPTH_SIZE, 32,
   AGL_DOUBLEBUFFER,
   AGL_ACCELERATED,
   AGL_NO_RECOVERY,
   AGL_NONE
};
static const GLint attrsFullscreen[] = 
{
   AGL_FULLSCREEN,
   AGL_RGBA,
   AGL_PIXEL_SIZE, 32,
   // AGL_DEPTH_SIZE, 32,
   AGL_DOUBLEBUFFER,
   AGL_ACCELERATED,
   AGL_NO_RECOVERY,
   AGL_SINGLE_RENDERER,
   AGL_NONE
};


char beginroutine(void);
char mainloop(void);
void endroutine(void);

#define DIM(arr)	(sizeof(arr) / sizeof(arr[0]))
static const EventTypeSpec keyEventTypes[] =
{
	{ kEventClassKeyboard, kEventRawKeyDown },
//	{ kEventClassKeyboard, kEventRawKeyRepeat },
	{ kEventClassKeyboard, kEventRawKeyUp },
	{ kEventClassKeyboard, kEventRawKeyModifiersChanged },
};
static const EventTypeSpec mouseEventTypes[] =
{
	{ kEventClassMouse, kEventMouseDown },
	{ kEventClassMouse, kEventMouseUp },
	{ kEventClassMouse, kEventMouseMoved },
	{ kEventClassMouse, kEventMouseDragged },
};
static const EventTypeSpec commandEventTypes[] =
{
   { kEventClassCommand, kEventCommandProcess },
};

pascal void DoFrame(EventLoopTimerRef theTimer, void* userData)
{
	enabler.do_frame();
}

// Bits		
//  0 - 15  Repeat count for the current message. Not cumulative.
// 16 - 23  OEM Scan code.
//      24  Extended key (e.g. right-hand ALT, CTRL keys on a 101-/102-key keyboard)
// 25 - 28  Reserved.
//      29  Context code. Always 0 for key down or key up.
//      30  Previous key state. Always 1 for key up; for key down, 1 if key was down before message sent, otherwise 0.
//      31  Transition state. Always 0 for key down, always 1 for key up.
static DWORD sKeyState[256] = { 0 };
static const DWORD kKeyIsUp = 0x80000000ul;
static const DWORD kKeyWasDown = 0x40000000ul;

static const char* sShiftedChars   = "~!@#$%^&*()_+{}|:\"<>?";
static const char* sUnshiftedChars = "`1234567890-=[]\\;',./";

struct KeyInfo { int vkey; char* name; };
#define VKEY(x)	{ VK_##x, "VK_" #x }
#define NOKEY	{ -1, "" }
static const KeyInfo keyCodeToVKey[128] = 
{
	// 0x00 - 0x0f
	NOKEY, NOKEY, NOKEY, NOKEY, NOKEY, NOKEY, NOKEY, NOKEY,
	NOKEY, NOKEY, NOKEY, NOKEY, NOKEY, NOKEY, NOKEY, NOKEY,
	// 0x10 - 0x1f
	NOKEY, NOKEY, NOKEY, NOKEY,
	NOKEY, NOKEY, NOKEY, NOKEY,
	{ 0xbb, "+ (plus)" },
	NOKEY, NOKEY,
	{ 0xbd, "- (minus)" },
	NOKEY, NOKEY, {0xdd, "] (right bracket)"}, NOKEY,
	// 0x20 - 0x2f
	NOKEY, {0xdb,"[ (left bracket)"}, NOKEY, NOKEY,
	VKEY(RETURN),
	NOKEY, NOKEY, {0xde,"\' (apostrophe)"},
	NOKEY, {0xba,"; (semicolon)"}, {0xdc,"\\ (backslash)"},
	{ 0xbc, ", (comma)" }, 
	{0xbf,"/ (forward slash)"}, NOKEY, NOKEY,
	{ 0xbe, ". (period)" }, 
	// 0x30 - 0x3f
	VKEY(TAB),
	VKEY(SPACE),
	{0xc0,"` (single quote)"},
	VKEY(BACK),
	NOKEY,
	VKEY(ESCAPE),
	NOKEY, NOKEY,
	NOKEY, NOKEY, NOKEY, NOKEY, 
	NOKEY, NOKEY, NOKEY, NOKEY, 
	// 0x40 - 0x4f
	NOKEY,
	VKEY(DECIMAL),
	NOKEY,
	VKEY(MULTIPLY),
	NOKEY,
	VKEY(ADD),
	NOKEY,
	VKEY(CLEAR),
	NOKEY,
	NOKEY,
	NOKEY,
	VKEY(DIVIDE),
	VKEY(SEPARATOR),	// numpad Enter
	NOKEY,
	VKEY(SUBTRACT),
	NOKEY,
	// 0x50 - 0x5f
	NOKEY,
	NOKEY,
	VKEY(NUMPAD0),
	VKEY(NUMPAD1),
	VKEY(NUMPAD2),
	VKEY(NUMPAD3),
	VKEY(NUMPAD4),
	VKEY(NUMPAD5),
	VKEY(NUMPAD6),
	VKEY(NUMPAD7),
	NOKEY,
	VKEY(NUMPAD8),
	VKEY(NUMPAD9),
	NOKEY,
	NOKEY,
	NOKEY,
	// 0x60 - 0x6f
	VKEY(F5),
	VKEY(F6),
	VKEY(F7),
	VKEY(F3),
	VKEY(F8),
	VKEY(F9),
	NOKEY,
	VKEY(F11),
	NOKEY,
	VKEY(F13),
	VKEY(F16),
	VKEY(F14),
	NOKEY,
	VKEY(F10),
	NOKEY,
	VKEY(F12),
	// 0x70 - 0x7f
	NOKEY,
	VKEY(F15),
	VKEY(HELP),
	VKEY(HOME),
	VKEY(PRIOR),
	VKEY(DELETE),
	VKEY(F4),
	VKEY(END),
	VKEY(F2),
	VKEY(NEXT),
	VKEY(F1),
	VKEY(LEFT),
	VKEY(RIGHT),
	VKEY(DOWN),
	VKEY(UP),
	NOKEY,
};
#undef VKEY
#undef NOKEY


static WPARAM SetKeyState(EventRef event, LPARAM& lParam)	// returns Windows virtual key
{
	UInt32 kcode, size;
	UniChar ch;
	
	bool isDown = (GetEventKind(event) == kEventRawKeyDown);
	GetEventParameter(event, kEventParamKeyCode, typeUInt32, NULL, sizeof(kcode), NULL, &kcode);
	GetEventParameter(event, kEventParamKeyUnicodes, typeUnicodeText, NULL, sizeof(ch), &size, &ch);
	
	// Convert to a windows key code.
	int vkey;
	if (kcode < 128 && keyCodeToVKey[kcode].vkey >= 0)
	{
		vkey = keyCodeToVKey[kcode].vkey;
		//if (isDown) fprintf(stderr, "keycode: 0x%02x => vkey: 0x%02x (%s)\n", 
		//	kcode, vkey, keyCodeToVKey[kcode].name);
	}
	else if ('A' <= ch && ch <= 'Z')
	{
		vkey = ch;
		//if (isDown) fprintf(stderr, "alpha key '%c' (0x%02x)\n", vkey, kcode);
	}
	else if ('a' <= ch && ch <= 'z')
	{
		vkey = ch - 'a' + 'A';
		//if (isDown) fprintf(stderr, "alpha key '%c' (0x%02x)\n", vkey, kcode);
	}
	else if ('0' <= ch && ch <= '9')
	{
		vkey = ch;
		//if (isDown) fprintf(stderr, "numeric key '%c' (0x%02x)\n", vkey, kcode);
	}
	else if (char* pch = strchr(sShiftedChars, ch))
	{
		vkey = sUnshiftedChars[pch - sShiftedChars];
		//if (isDown) fprintf(stderr, "shifted key '%c' becomes '%c' (0x%02x)\n", ch, vkey, kcode);
	}
	else
	{
		vkey = 0;
		//if (isDown) fprintf(stderr, "Unknown key code: 0x%02x (%d)\n", kcode, ch);
	}
	
	lParam  = 1;
	lParam |= (kcode & 0x000000fful) << 16;
	if (isDown)
	{
		if ((sKeyState[vkey] & kKeyIsUp) == 0)
			lParam |= kKeyWasDown;
	}
	else
	{
		lParam |= (kKeyIsUp | kKeyWasDown);
	}
	
	sKeyState[vkey] = lParam;
	return vkey;
}

OSStatus HandleKeyboard(EventHandlerCallRef caller, EventRef event, void* userData)
{
	GL_Window* window = (GL_Window*) userData;
	
	assert(GetEventClass(event) == kEventClassKeyboard);

	UInt32 kind = GetEventKind(event);
	if (kind == kEventRawKeyModifiersChanged)
	{
		static UInt32 lastMods = 0;
		UInt32 mods;
		GetEventParameter(event, kEventParamKeyModifiers, typeUInt32, NULL, sizeof(mods), NULL, &mods);
		UInt32 changed = (lastMods ^ mods);
		lastMods = mods;
		//fprintf(stderr, "Modifiers: %08x (changed: %08x)\n", mods, changed);
		
		// Mac OS X can't distunguish between left/right shift, etc?
		// So it seems, as commented in <Events.h>
			
		const int modsToCheck[] = { shiftKey, alphaLock, optionKey, controlKey, 0 };
		const int modsToVKey[]  = { VK_SHIFT, VK_CAPITAL, VK_MENU, VK_CONTROL, -1 };
		
		bool isDown;
		WPARAM wParam;
		LPARAM lParam;

		int x = 0;
		while (modsToCheck[x])
		{
			if (changed & modsToCheck[x])
			{
				wParam = modsToVKey[x];
				isDown = (mods & modsToCheck[x]);
				lParam = 1;
				if (isDown)
				{
					if ((sKeyState[wParam] & kKeyIsUp) == 0)
						lParam |= kKeyWasDown;
				}
				else
				{
					lParam |= (kKeyIsUp | kKeyWasDown);
				}
				sKeyState[wParam] = lParam;

				if (isDown)
				{
					if (!(lParam & BIT31))
					{
						enabler_inputst newi;
							newi.key=wParam;
							// The !! were added because newi.shift is a char while
							// GetKeyState returns a short. Not sure if/how/why this
							// works on Windows.
							newi.shift=!!(GetKeyState(VK_SHIFT)>>1);
							newi.caps=!!GetKeyState(VK_CAPITAL);
							//fprintf(stderr, "shift:%d  capital:%d\n", newi.shift, newi.caps);
						enabler.add_input(newi);
					}
					window->keys->keyDown[wParam] = TRUE;
				}
				
				if (!isDown)
				{
					window->keys->keyDown[wParam] = FALSE;
				}
			}
			
			++x;
		}
		
		return noErr;
	}
	else	// kEventRawKeyDown|Up
	{
		WPARAM wParam;
		LPARAM lParam;
		wParam = SetKeyState(event, lParam);
		
		switch (kind)
		{
			case kEventRawKeyDown:
				if (/*0 <= wParam &&*/ wParam <= 255)
				{
					if (!(lParam & BIT31))
					{
						enabler_inputst newi;
							newi.key=wParam;
							newi.shift=GetKeyState(VK_SHIFT)>>1;
							newi.caps=GetKeyState(VK_CAPITAL);
						enabler.add_input(newi);
					}
					window->keys->keyDown[wParam] = TRUE;
					return noErr;
				}
				break;
				
			case kEventRawKeyUp:
				if (/*0 <= wParam &&*/ wParam <= 255)
				{
					window->keys->keyDown[wParam] = FALSE;
					return noErr;
				}
				break;

			default:
				break;
		}
	}
	
	return eventNotHandledErr;
}

static void AddToggleFullscreenKey(GL_Window* window)
{
   enabler_inputst newi;
   newi.key=VK_F11;
   newi.shift=GetKeyState(VK_SHIFT)>>1;
   newi.caps=GetKeyState(VK_CAPITAL);
   enabler.add_input(newi);
   window->keys->keyDown[VK_F11] = TRUE;
}

OSStatus HandleCommand(EventHandlerCallRef caller, EventRef event, void* userData)
{
   GL_Window* window = (GL_Window*) userData;
   
   HICommand cmd;
   
   GetEventParameter(event, kEventParamDirectObject, typeHICommand, NULL,
                     sizeof(HICommand), NULL, &cmd);
   
   if (cmd.commandID == 'full')
   {
      AddToggleFullscreenKey(window);
      return noErr;
   }
   else if (cmd.commandID == kHICommandQuit)
   {
      // Uncomment if you want to do this the KQ way: force the users to exit
      // cleanly via a menu option in the game. While commented out, users can
      // hit Cmd-Q (standard Mac shortcut) to exit without saving. In the future,
      // we could do more here, like prompt the user if they really want to quit,
      // but at the moment, this is easiest (until someone complains).
      
      //return noErr;
   }
      
   return eventNotHandledErr;
}

static bool sMouseWasInWindow = false;
OSStatus HandleMouse(EventHandlerCallRef caller, EventRef event, void* userData)
{
	GL_Window* window = (GL_Window*) userData;

	assert(GetEventClass(event) == kEventClassMouse);

	EventMouseButton button;
	Point pt;
	
	switch (GetEventKind(event))
	{
		case kEventMouseDown:
			GetEventParameter(event, kEventParamMouseButton, typeMouseButton, NULL, sizeof(button), NULL, &button);
			switch (button)
			{
				case kEventMouseButtonPrimary:    enabler.mouse_lbut=1;  break;
				case kEventMouseButtonSecondary:  enabler.mouse_rbut=1;  break;
				default: break;
			}
			return eventNotHandledErr;	// must allow mouse down to propogate to window
			
		case kEventMouseUp:
			GetEventParameter(event, kEventParamMouseButton, typeMouseButton, NULL, sizeof(button), NULL, &button);
			switch (button)
			{
				case kEventMouseButtonPrimary:    enabler.mouse_lbut=0;enabler.mouse_lbut_lift=1;  break;
				case kEventMouseButtonSecondary:  enabler.mouse_rbut=0;enabler.mouse_rbut_lift=1;  break;
				default: break;
			}
			return noErr;
			
		case kEventMouseMoved:
		case kEventMouseDragged:
			GetEventParameter(event, kEventParamMouseLocation, typeQDPoint, NULL, sizeof(pt), NULL, &pt);
			if (!window->init.isFullScreen)
			{
				Rect cr;
				GetWindowBounds(window->wnd, kWindowContentRgn, &cr);
				if (pt.v < cr.top || pt.v > cr.bottom || pt.h < cr.left || pt.h > cr.right)
				{
					if (sMouseWasInWindow)
					{
						CGDisplayShowCursor(kCGDirectMainDisplay);
						sMouseWasInWindow = false;
					}
					// mouse exited window
					enabler.oldmouse_x=-1;
					enabler.oldmouse_y=-1;
					enabler.mouse_x=-1;
					enabler.mouse_y=-1;
					enabler.mouse_lbut=0;
					enabler.mouse_rbut=0;
					enabler.mouse_lbut_lift=0;
					enabler.mouse_rbut_lift=0;
					enabler.tracking_on=0;
					return eventNotHandledErr;
				}
				else if (!sMouseWasInWindow)
				{
					CGDisplayHideCursor(kCGDirectMainDisplay);
					sMouseWasInWindow = true;
				}
				
				HIPoint hipt = { pt.h, pt.v };
				HIViewRef srcView = HIViewGetRoot(window->wnd);
				HIViewRef dstView;
				HIViewFindByID(srcView, kHIViewWindowContentID, &dstView);
				
				if (HIPointConvert != NULL)
				{
					HIPointConvert(&hipt, kHICoordSpaceScreenPixel, NULL,
						kHICoordSpaceView, dstView);
				}
				else
				{
					hipt.x -= cr.left;
					hipt.y -= cr.top;
					HIViewConvertPoint(&hipt, NULL, dstView);
				}
				
				pt.h = (short) hipt.x;
				pt.v = (short) hipt.y;
			}
			
			enabler.oldmouse_x=enabler.mouse_x;
			enabler.oldmouse_y=enabler.mouse_y;
			enabler.mouse_x=(long)pt.h;
			enabler.mouse_y=(long)pt.v;
			enabler.tracking_on = 1;
			return noErr;
			
		default:
			break;
	}
	
	return eventNotHandledErr;
}

int enablerst::loop(HINSTANCE hInstance)
{
	application.mainNib = NULL;

	// Fill Out Window
	ZeroMemory (&window, sizeof (GL_Window));							// Make Sure Memory Is Zeroed
	window.keys					= &keys;								// Window Key Structure
	window.init.application		= &application;							// Window Application
	window.init.title			= GAME_TITLE_STRING;					// Window Title
	window.init.bitsPerPixel	= 32;									// Bits Per Pixel
	window.init.isFullScreen	= TRUE;									// Fullscreen? (Set To TRUE)
	ZeroMemory (&keys, sizeof (Keys));									// Zero keys Structure

	// Register A Class For Our Window To Use
	if (!register_window_class (&application))
	{
		MessageBox (HWND_DESKTOP, "Error Registering Window Class!", "Error", MB_OK | MB_ICONEXCLAMATION);
		return -1;														// Terminate Application
		}

	is_program_looping = TRUE;
	if (!beginroutine())
	{
		is_program_looping = FALSE;
	}
	else
	{
		QueryPerformanceCounter(&qpc);
		
		window.init.isFullScreen = create_full_screen;
		if (!create_window_GL(&window))
		{
			MessageBox (HWND_DESKTOP, "Error Creating OpenGL Window", "Error", MB_OK | MB_ICONEXCLAMATION);
			is_program_looping = FALSE;
		}
		else
		{
			// At This Point We Should Have A Window That Is Setup To Render OpenGL
			enabler.create_textures();
			
			EventHandlerUPP keyHandlerUPP = NewEventHandlerUPP(HandleKeyboard);
			InstallApplicationEventHandler(keyHandlerUPP,
								DIM(keyEventTypes), keyEventTypes, &window, NULL);
								
			EventHandlerUPP mouseHandlerUPP = NewEventHandlerUPP(HandleMouse);
			InstallApplicationEventHandler(mouseHandlerUPP,
								DIM(mouseEventTypes), mouseEventTypes, &window, NULL);
         
         EventHandlerUPP commandHandlerUPP = NewEventHandlerUPP(HandleCommand);
         InstallApplicationEventHandler(commandHandlerUPP,
                  DIM(commandEventTypes), commandEventTypes, &window, NULL);
			
			EventLoopRef mainLoop = GetMainEventLoop();
			EventLoopTimerUPP timerUPP = NewEventLoopTimerUPP(DoFrame);
			EventLoopTimerRef theTimer;
			InstallEventLoopTimer(mainLoop, 0, 0.01, timerUPP, NULL, &theTimer);
         
			RunApplicationEventLoop();
			
			RemoveEventLoopTimer(theTimer);
			DisposeEventLoopTimerUPP(timerUPP);

         DisposeEventHandlerUPP(commandHandlerUPP);
			DisposeEventHandlerUPP(mouseHandlerUPP);
			DisposeEventHandlerUPP(keyHandlerUPP);

			enabler.remove_textures();

			destroy_window_GL(&window);									// Destroy The Active Window
		}
	}
	
	endroutine();

	DisposeNibReference(application.mainNib);
	application.mainNib = NULL;

	return 0;
}

void enablerst::do_frame()
{
	long advanced=0;
	int t;
	for(t=0;t<tile.size();t++)tile[t]->flag|=TILEFLAG_DEAD;
	if(tile.size()>0)next_tile_slot=0;
	else next_tile_slot=-1;

	if(mainloop())
		{
		//PostMessage(window.hWnd, WM_QUIT, 0, 0);	// Send A WM_QUIT Message
		is_program_looping = FALSE;					// Stop Looping Of The Program
		}
	else
		{
		advanced++;
		//QueryPerformanceCounter(&qpc);
		}

	if(!is_program_looping)
		{
		QuitApplicationEventLoop();
		return;
		}

	if(advanced)
		{
		render(window);
		
		current_render_count++;
		secondary_render_count++;
		}
}

void enablerst::load_bitmap_header(const string &filename, BITMAPINFOHEADER &bitmapInfoHeader)
{
	FILE *filePtr;                        // the file pointer
	BITMAPFILEHEADER  bitmapFileHeader;   // bitmap file header

	// open filename in "read binary" mode
	filePtr = fopen(filename.c_str(), "rb");
	if (filePtr == NULL)return;

	// read the bitmap file header
	fread(&bitmapFileHeader, sizeof(BITMAPFILEHEADER), 1, filePtr);
	bitmapFileHeader.bfType = byteswap(bitmapFileHeader.bfType);
	bitmapFileHeader.bfSize = byteswap(bitmapFileHeader.bfSize);
	bitmapFileHeader.bfReserved1 = byteswap(bitmapFileHeader.bfReserved1);
	bitmapFileHeader.bfReserved2 = byteswap(bitmapFileHeader.bfReserved2);
	bitmapFileHeader.bfOffBits = byteswap(bitmapFileHeader.bfOffBits);

	// verify that this is a bitmap by checking for the universal bitmap id
	if (bitmapFileHeader.bfType != BITMAP_ID)
		{
		fclose(filePtr);
		return;
		}

	// read the bitmap information header
	fread(&bitmapInfoHeader, sizeof(BITMAPINFOHEADER), 1, filePtr);
	bitmapInfoHeader.biSize = byteswap(bitmapInfoHeader.biSize);
	bitmapInfoHeader.biWidth = byteswap(bitmapInfoHeader.biWidth);
	bitmapInfoHeader.biHeight = byteswap(bitmapInfoHeader.biHeight);
	bitmapInfoHeader.biPlanes = byteswap(bitmapInfoHeader.biPlanes);
	bitmapInfoHeader.biBitCount = byteswap(bitmapInfoHeader.biBitCount);
	bitmapInfoHeader.biCompression = byteswap(bitmapInfoHeader.biCompression);
	bitmapInfoHeader.biSizeImage = byteswap(bitmapInfoHeader.biSizeImage);
	bitmapInfoHeader.biXPelsPerMeter = byteswap(bitmapInfoHeader.biXPelsPerMeter);
	bitmapInfoHeader.biYPelsPerMeter = byteswap(bitmapInfoHeader.biYPelsPerMeter);
	bitmapInfoHeader.biClrUsed = byteswap(bitmapInfoHeader.biClrUsed);
	bitmapInfoHeader.biClrImportant = byteswap(bitmapInfoHeader.biClrImportant);
	bitmapInfoHeader.biSizeImage=bitmapInfoHeader.biWidth*bitmapInfoHeader.biHeight*3;

	// close the file
	fclose(filePtr);
}

//TARN: don't remember where this is from -- an OpenGL book sample probably
/*
 LoadBitmapFile()

 Returns a pointer to the bitmap image of the bitmap specified by filename.
 Also returns the bitmap header information. No support for 8-bit bitmaps.
*/
unsigned char *enablerst::load_bitmap_file(const string &filename, BITMAPINFOHEADER *bitmapInfoHeader)
{
  FILE *filePtr;                        // the file pointer
  BITMAPFILEHEADER  bitmapFileHeader;   // bitmap file header
  unsigned char    *bitmapImage;        // bitmap image data
  unsigned int      imageIdx = 0;       // image index counter
  unsigned char     tempRGB;            // swap variable

  // open filename in "read binary" mode
  filePtr = fopen(filename.c_str(), "rb");
  if (filePtr == NULL)
    return NULL;

  // read the bitmap file header
  fread(&bitmapFileHeader, sizeof(BITMAPFILEHEADER), 1, filePtr);
	bitmapFileHeader.bfType = byteswap(bitmapFileHeader.bfType);
	bitmapFileHeader.bfSize = byteswap(bitmapFileHeader.bfSize);
	bitmapFileHeader.bfReserved1 = byteswap(bitmapFileHeader.bfReserved1);
	bitmapFileHeader.bfReserved2 = byteswap(bitmapFileHeader.bfReserved2);
	bitmapFileHeader.bfOffBits = byteswap(bitmapFileHeader.bfOffBits);

  // verify that this is a bitmap by checking for the universal bitmap id
  if (bitmapFileHeader.bfType != BITMAP_ID)
  {
    fclose(filePtr);
    return NULL;
  }

  // read the bitmap information header
  fread(bitmapInfoHeader, sizeof(BITMAPINFOHEADER), 1, filePtr);
	bitmapInfoHeader->biSize = byteswap(bitmapInfoHeader->biSize);
	bitmapInfoHeader->biWidth = byteswap(bitmapInfoHeader->biWidth);
	bitmapInfoHeader->biHeight = byteswap(bitmapInfoHeader->biHeight);
	bitmapInfoHeader->biPlanes = byteswap(bitmapInfoHeader->biPlanes);
	bitmapInfoHeader->biBitCount = byteswap(bitmapInfoHeader->biBitCount);
	bitmapInfoHeader->biCompression = byteswap(bitmapInfoHeader->biCompression);
	bitmapInfoHeader->biSizeImage = byteswap(bitmapInfoHeader->biSizeImage);
	bitmapInfoHeader->biXPelsPerMeter = byteswap(bitmapInfoHeader->biXPelsPerMeter);
	bitmapInfoHeader->biYPelsPerMeter = byteswap(bitmapInfoHeader->biYPelsPerMeter);
	bitmapInfoHeader->biClrUsed = byteswap(bitmapInfoHeader->biClrUsed);
	bitmapInfoHeader->biClrImportant = byteswap(bitmapInfoHeader->biClrImportant);
  bitmapInfoHeader->biSizeImage=bitmapInfoHeader->biWidth*bitmapInfoHeader->biHeight*3;

  // move file pointer to beginning of bitmap data
  fseek(filePtr, bitmapFileHeader.bfOffBits, SEEK_SET);

	// allocate enough memory for the bitmap image data
	if(bitmapInfoHeader->biSizeImage>0)
		{
		bitmapImage=new unsigned char[bitmapInfoHeader->biSizeImage];
		}
	else bitmapImage=NULL;

  // verify memory allocation
  if (!bitmapImage)
  {
    delete[] bitmapImage;
    fclose(filePtr);
    return NULL;
  }

  // read in the bitmap image data
  fread(bitmapImage, 1, bitmapInfoHeader->biSizeImage, filePtr);

  // make sure bitmap image data was read
  if (bitmapImage == NULL)
  {
    fclose(filePtr);
    return NULL;
  }

  // swap the R and B values to get RGB since the bitmap color format is in BGR
  for (imageIdx = 0; imageIdx < bitmapInfoHeader->biSizeImage; imageIdx+=3)
  {
    tempRGB = bitmapImage[imageIdx];
    bitmapImage[imageIdx] = bitmapImage[imageIdx + 2];
    bitmapImage[imageIdx + 2] = tempRGB;
  }

  // close the file and return the bitmap image data
  fclose(filePtr);
  return bitmapImage;
} // end LoadBitmapFile()


//TARN: don't remember where this is from -- an OpenGL book sample probably
/*
 LoadBitmapFileWithAlpha

 Loads a bitmap file normally, and then adds an alpha component to use for
 blending
*/
unsigned char *enablerst::load_bitmap_file_with_alpha(const string &filename, BITMAPINFOHEADER *bitmapInfoHeader)
{
  unsigned char *bitmapImage = load_bitmap_file(filename, bitmapInfoHeader);

  if(bitmapImage==NULL)return NULL;

  unsigned char *bitmapWithAlpha=new unsigned char[bitmapInfoHeader->biSizeImage * 4 / 3];

  if (bitmapImage == NULL || bitmapWithAlpha == NULL)
    return NULL;

  // loop through the bitmap data
  for (unsigned int src = 0, dst = 0; src < bitmapInfoHeader->biSizeImage; src +=3, dst +=4)
  {
    // if the pixel is magenta, set the alpha to 0. Otherwise, set it to 255.
    if (bitmapImage[src] == 255 && bitmapImage[src+1] == 0 && bitmapImage[src+2] == 255)
      bitmapWithAlpha[dst+3] = 0;
    else
      bitmapWithAlpha[dst+3] = 0xFF;

    // copy pixel data over
    bitmapWithAlpha[dst] = bitmapImage[src];
    bitmapWithAlpha[dst+1] = bitmapImage[src+1];
    bitmapWithAlpha[dst+2] = bitmapImage[src+2];
  }

  delete[] bitmapImage;

  return bitmapWithAlpha;
} // end LoadBitmapFileWithAlpha()

void enablerst::save_texture_data_to_bmp(long pos,string &filename)
{
	if(pos<0&&pos>=texture_data.size())return;

  FILE *filePtr;
  BITMAPFILEHEADER bitmapFileHeader;
  BITMAPINFOHEADER bitmapInfoHeader;
  unsigned char    *bitmapImage=enabler.get_texture_data(pos);
  unsigned int      imageIdx=0;       // image index counter
  unsigned char     tempRGB;            // swap variable

  // open filename in "write binary" mode
  filePtr = fopen(filename.c_str(), "wb");
  if (filePtr == NULL)
    return;

  bitmapFileHeader.bfOffBits=sizeof(BITMAPFILEHEADER)+sizeof(BITMAPINFOHEADER);
  bitmapFileHeader.bfSize=sizeof(BITMAPFILEHEADER);
  bitmapFileHeader.bfType=BITMAP_ID;
  bitmapFileHeader.bfReserved1=0;
  bitmapFileHeader.bfReserved2=0;

  // write the bitmap file header
	bitmapFileHeader.bfType = byteswap(bitmapFileHeader.bfType);
	bitmapFileHeader.bfSize = byteswap(bitmapFileHeader.bfSize);
	bitmapFileHeader.bfReserved1 = byteswap(bitmapFileHeader.bfReserved1);
	bitmapFileHeader.bfReserved2 = byteswap(bitmapFileHeader.bfReserved2);
	bitmapFileHeader.bfOffBits = byteswap(bitmapFileHeader.bfOffBits);
  fwrite(&bitmapFileHeader, sizeof(BITMAPFILEHEADER), 1, filePtr);

  bitmapInfoHeader.biBitCount=24+((int)texture_data[pos]->alpha_texture)*8;
  bitmapInfoHeader.biClrImportant=0;
  bitmapInfoHeader.biClrUsed=0;
  bitmapInfoHeader.biCompression=0;
  bitmapInfoHeader.biWidth=texture_data[pos]->width;
  bitmapInfoHeader.biHeight=texture_data[pos]->height;
  long bytes=bitmapInfoHeader.biBitCount/8;
  bitmapInfoHeader.biPlanes=1;
  bitmapInfoHeader.biSize=sizeof(BITMAPINFOHEADER);
  bitmapInfoHeader.biSizeImage=bitmapInfoHeader.biHeight*bitmapInfoHeader.biWidth*bytes;
  bitmapInfoHeader.biXPelsPerMeter=1;
  bitmapInfoHeader.biYPelsPerMeter=1;

	bitmapInfoHeader.biSize = byteswap(bitmapInfoHeader.biSize);
	bitmapInfoHeader.biWidth = byteswap(bitmapInfoHeader.biWidth);
	bitmapInfoHeader.biHeight = byteswap(bitmapInfoHeader.biHeight);
	bitmapInfoHeader.biPlanes = byteswap(bitmapInfoHeader.biPlanes);
	bitmapInfoHeader.biBitCount = byteswap(bitmapInfoHeader.biBitCount);
	bitmapInfoHeader.biCompression = byteswap(bitmapInfoHeader.biCompression);
	bitmapInfoHeader.biSizeImage = byteswap(bitmapInfoHeader.biSizeImage);
	bitmapInfoHeader.biXPelsPerMeter = byteswap(bitmapInfoHeader.biXPelsPerMeter);
	bitmapInfoHeader.biYPelsPerMeter = byteswap(bitmapInfoHeader.biYPelsPerMeter);
	bitmapInfoHeader.biClrUsed = byteswap(bitmapInfoHeader.biClrUsed);
	bitmapInfoHeader.biClrImportant = byteswap(bitmapInfoHeader.biClrImportant);
	bitmapInfoHeader.biSizeImage = byteswap(bitmapInfoHeader.biSizeImage);


  // write the bitmap information header
  fwrite(&bitmapInfoHeader, sizeof(BITMAPINFOHEADER), 1, filePtr);

  // swap the R and B values to get RGB since the bitmap color format is in BGR
  for (imageIdx = 0; imageIdx < bitmapInfoHeader.biSizeImage; imageIdx+=bytes)
  {
    tempRGB = bitmapImage[imageIdx];
    bitmapImage[imageIdx] = bitmapImage[imageIdx + 2];
    bitmapImage[imageIdx + 2] = tempRGB;
  }

  // read in the bitmap image data
  fwrite(bitmapImage, 1, bitmapInfoHeader.biSizeImage, filePtr);

  // close the file
  fclose(filePtr);
}

void enablerst::render(GL_Window &window)
{
	if(flag & ENABLERFLAG_RENDER)
		{
		//CLEAR
		glClearColor(clear_r,clear_g,clear_b,clear_a);
		glClear(GL_COLOR_BUFFER_BIT);

		//DRAW EVERYTHING TO BACK BUFFER
		render_tiles();

		//DO BUFFERS
		aglSwapBuffers (window.context);

		//DON'T PRINT AGAIN UNTIL ASKED
		flag&=~ENABLERFLAG_RENDER;
		}
}

void enablerst::terminate_application(GL_Window* window)
{
// TODO	-- NOTHING ACTUALLY USING THIS?
//	PostMessage(window->hWnd, WM_QUIT, 0, 0);	// Send A WM_QUIT Message
	is_program_looping = FALSE;					// Stop Looping Of The Program
}

void enablerst::toggle_fullscreen(GL_Window* window)
{
	enabler.create_full_screen = !enabler.create_full_screen;
   
   enabler.remove_textures();
   
	destroy_window_GL(window);
	window->init.isFullScreen = create_full_screen;
	if (!create_window_GL(window))
	{
		MessageBox (HWND_DESKTOP, "Error Creating OpenGL Window", "Error", MB_OK | MB_ICONEXCLAMATION);
		is_program_looping = FALSE;
		return;
	}
   
   enabler.create_textures();
	render(*window);
}

void enablerst::reshape_GL(int width,int height)				// Reshape The Window When It's Moved Or Resized
{
	glDisable(GL_ALPHA_TEST); glDisable(GL_BLEND); glDisable(GL_DEPTH_TEST);
	glDisable(GL_DITHER); glDisable(GL_FOG); glDisable(GL_LIGHTING);
	glDisable(GL_LOGIC_OP); glDisable(GL_STENCIL_TEST);
	glDisable(GL_TEXTURE_1D);glShadeModel(GL_FLAT);
	glDisable(GL_TEXTURE_2D); glPixelTransferi(GL_MAP_COLOR, GL_FALSE);
	glPixelTransferi(GL_RED_SCALE, 1); glPixelTransferi(GL_RED_BIAS, 0);
	glPixelTransferi(GL_GREEN_SCALE, 1); glPixelTransferi(GL_GREEN_BIAS, 0);
	glPixelTransferi(GL_BLUE_SCALE, 1); glPixelTransferi(GL_BLUE_BIAS, 0);
	glPixelTransferi(GL_ALPHA_SCALE, 1); glPixelTransferi(GL_ALPHA_BIAS, 0);

	window_width=width;
	window_height=height;

	glViewport (0, 0, (GLsizei)(width), (GLsizei)(height));		// Reset The Current Viewport
	glMatrixMode (GL_PROJECTION);								// Select The Projection Matrix
	glLoadIdentity ();											// Reset The Projection Matrix
	glOrtho(0.0f, width, height, 0, -1.0, 1.0);
	glMatrixMode (GL_MODELVIEW);								// Select The Modelview Matrix
	glLoadIdentity ();											// Reset The Modelview Matrix
}

char enablerst::change_screen_resolution (int width, int height, int bitsPerPixel)	// Change The Screen Resolution
{
// TODO
	return false;
/*
	DEVMODE dmScreenSettings;											// Device Mode
	ZeroMemory (&dmScreenSettings, sizeof (DEVMODE));					// Make Sure Memory Is Cleared
	dmScreenSettings.dmSize				= sizeof (DEVMODE);				// Size Of The Devmode Structure
	dmScreenSettings.dmPelsWidth		= width;						// Select Screen Width
	dmScreenSettings.dmPelsHeight		= height;						// Select Screen Height
	dmScreenSettings.dmBitsPerPel		= bitsPerPixel;					// Select Bits Per Pixel
	dmScreenSettings.dmFields			= DM_BITSPERPEL | DM_PELSWIDTH | DM_PELSHEIGHT;
	if (ChangeDisplaySettings (&dmScreenSettings, CDS_FULLSCREEN) != DISP_CHANGE_SUCCESSFUL)
	{
		return FALSE;													// Display Change Failed, Return False
	}
	return TRUE;														// Display Change Was Successful, Return True
*/
}

char enablerst::create_window_GL(GL_Window* window)									// This Code Creates Our OpenGL Window
{
	OSStatus err = CreateWindowFromNib(window->init.application->mainNib, CFSTR("MainWindow"), &window->wnd);
	if (err != 0)
		return false;

	CFStringRef title = CFStringCreateWithCStringNoCopy(NULL, window->init.title, kCFStringEncodingWindowsLatin1, kCFAllocatorNull);
	SetWindowTitleWithCFString(window->wnd, title);
	CFRelease(title);

	if (window->init.isFullScreen)
	{
		window->init.width=desired_fullscreen_width;
		window->init.height=desired_fullscreen_height;
	}
	else
	{
		window->init.width=desired_windowed_width;
		window->init.height=desired_windowed_height;
	}
	Rect windowRect = {50, 50, 50+window->init.height, 50+window->init.width};	// TLBR
	SetWindowBounds(window->wnd, kWindowContentRgn, &windowRect);
	
   AGLPixelFormat pfFullscreen = NULL;
	if (window->init.isFullScreen)
	{
      GDHandle gdhDisplay;
      DMGetGDeviceByDisplayID((DisplayIDType)kCGDirectMainDisplay, &gdhDisplay, true);

      pfFullscreen = aglChoosePixelFormat(&gdhDisplay, 1, attrsFullscreen);
      if (!pfFullscreen)
         goto create_win_fail;

      contextFullscreen = aglCreateContext(pfFullscreen, NULL);//contextWindowed);
      if (!contextFullscreen) goto create_win_fail;	
      aglEnable(contextFullscreen, AGL_FS_CAPTURE_SINGLE);

      aglDestroyPixelFormat(pfFullscreen);
      pfFullscreen = NULL;
      
      window->context = contextFullscreen;
      
      fprintf(stderr, "aglSetFullScreen %d x %d\n", window->init.width, window->init.height);
      if (!aglSetFullScreen(window->context, window->init.width, window->init.height, 0, 0) ||
          !aglSetCurrentContext(window->context))
      {
         fprintf(stderr, "GL Error: %d\n", aglGetError());
         goto create_win_fail;
      }
	}
	else
	{
      window->context = contextWindowed;
		if (!aglSetDrawable(window->context, GetWindowPort(window->wnd)) ||
          !aglSetCurrentContext(window->context))
      {
         goto create_win_fail;
      }
      
      ShowWindow(window->wnd);
      SelectWindow(window->wnd);
	}

	reshape_GL(window->init.width, window->init.height);
	
	window->lastTickCount = GetTickCount();
	return true;
   
create_win_fail:
   destroy_window_GL(window);
   return false;
}

char enablerst::destroy_window_GL(GL_Window* window)								// Destroy The OpenGL Window & Release Resources
{
   aglSetDrawable(window->context, NULL);
   
   if(contextFullscreen)
   {  // we were in full screen mode
      aglSetCurrentContext(NULL);
      aglDestroyContext(contextFullscreen);
   }
   else
      aglSetCurrentContext(NULL);

	window->context = NULL;
	ReleaseWindow(window->wnd);
	window->wnd = NULL;
	return true;
}

//**************************

//TARN: I wrote everything below here (aside from port contributions)

void bitmap_class::clean(void)
{
	if(data!=NULL)
		{
		delete[] data;
		data=NULL;
		}
	glDeleteTextures(1,(GLuint*)&gennum);
	gennum=0;
}

bitmap_class::~bitmap_class()
{
	clean();
}

void enablerst::add_input(enabler_inputst &ninput)
{
	int i;
	for(i=0;i<100;i++)
		{
		if(input[i].key==0)
			{
			input[i]=ninput;
			break;
			}
		}
}

enablerst::enablerst()
{
	buffer_draw=0;
	center_x=0;center_y=0;

	tracking_on=0;
	oldmouse_x=-1;oldmouse_y=-1;mouse_x=-1;mouse_y=-1;
	mouse_lbut=0;mouse_rbut=0;
	mouse_lbut_lift=0;mouse_rbut_lift=0;

	set_color(1,1,1);

	desired_windowed_width=640;
	desired_windowed_height=300;
	desired_fullscreen_width=640;
	desired_fullscreen_height=300;

	ccolor[0][0]=0;
	ccolor[0][1]=0;
	ccolor[0][2]=0;
	ccolor[1][0]=0;
	ccolor[1][1]=0;
	ccolor[1][2]=128;
	ccolor[2][0]=0;
	ccolor[2][1]=128;
	ccolor[2][2]=0;
	ccolor[3][0]=0;
	ccolor[3][1]=128;
	ccolor[3][2]=128;
	ccolor[4][0]=128;
	ccolor[4][1]=0;
	ccolor[4][2]=0;
	ccolor[5][0]=128;
	ccolor[5][1]=0;
	ccolor[5][2]=128;
	ccolor[6][0]=128;
	ccolor[6][1]=128;
	ccolor[6][2]=0;
	ccolor[7][0]=192;
	ccolor[7][1]=192;
	ccolor[7][2]=192;
	ccolor[8][0]=128;
	ccolor[8][1]=128;
	ccolor[8][2]=128;
	ccolor[9][0]=0;
	ccolor[9][1]=0;
	ccolor[9][2]=255;
	ccolor[10][0]=0;
	ccolor[10][1]=255;
	ccolor[10][2]=0;
	ccolor[11][0]=0;
	ccolor[11][1]=255;
	ccolor[11][2]=255;
	ccolor[12][0]=255;
	ccolor[12][1]=0;
	ccolor[12][2]=0;
	ccolor[13][0]=255;
	ccolor[13][1]=0;
	ccolor[13][2]=255;
	ccolor[14][0]=255;
	ccolor[14][1]=255;
	ccolor[14][2]=0;
	ccolor[15][0]=255;
	ccolor[15][1]=255;
	ccolor[15][2]=255;

	next_tile_slot=-1;

	next_gridrect_id=1;
	next_cursesrect_id=1;
	next_font_id=1;
	flag=0;
	QueryPerformanceFrequency(&qpfr);
	qprate.QuadPart=qpfr.QuadPart/100;
}

void enablerst::create_textures(void)
{
	int t;
	for(t=(int)font.size()-1;t>=0;t--)
		{
		font[t]->create_textures(font[t]->fontfile,font[t]->lx,font[t]->ly);
		}
	for(t=(int)texture.size()-1;t>=0;t--)
		{
		if(texture[t]==NULL)continue;

		texture[t]->create_texture();
		}
	for(t=(int)cursesrect.size()-1;t>=0;t--)
		{
		cursesrect[t]->adopt_textures(cursesrect[t]->font_id);
		}
}

void fontst::remove_textures(void)
{
	glDeleteTextures(256,(GLuint*)gennum);
}

void texturest::remove_texture(void)
{
	old_gennum=gennum;
	glDeleteTextures(1,(GLuint*)&gennum);
}

void fontst::create_textures(const string &filename,long letx,long lety)
{
	fontfile=filename;
	if(letx<=0||lety<=0)return;

	bitmap_class bmp;
		bmp.data=enabler.load_bitmap_file_with_alpha(filename,&bmp.info);

		letsizex=bmp.info.biWidth/letx;
		letsizey=bmp.info.biHeight/lety;
		powsizex=letsizex;
		powsizey=letsizey;
		int t=0;
		for(t=0;t<8;t++)
			{
			if(powsizex<=(1<<t)){powsizex=(1<<t);break;}
			}
		for(t=0;t<8;t++)
			{
			if(powsizey<=(1<<t)){powsizey=(1<<t);break;}
			}

		if(bmp.data!=NULL)
			{
			char *glyph=new char[powsizex*powsizey*4];

			memset(gennum,0,sizeof(unsigned int)*256);
			glGenTextures(256,(GLuint*)gennum);

			int x,y,sx,sy;
			int g;
			int d;

			for(g=0;g<256;g++)
				{
				memset(glyph,0,powsizex*powsizey*4*sizeof(char));
				sx=(letx-1-(g%letx))*letsizex;
				sy=(g/letx)*letsizey;
				d=0;

				if(var_font)
					{
					letwidth[255-g]=1;
					}
				else letwidth[255-g]=letsizex;

				if(g=='R')
					{
					int a;
					a=0;
					}

				for(y=sy;y<sy+letsizey;y++)
					{
					for(x=sx;x<sx+letsizex;x++,d+=4)
						{
						glyph[d]=bmp.data[(x+y*bmp.info.biWidth)*4];
						glyph[d+1]=bmp.data[(x+y*bmp.info.biWidth)*4+1];
						glyph[d+2]=bmp.data[(x+y*bmp.info.biWidth)*4+2];
						glyph[d+3]=bmp.data[(x+y*bmp.info.biWidth)*4+3];
						if(glyph[d+3]!=0&&var_font&&x-sx+2>letwidth[255-g])
							{
							letwidth[255-g]=x-sx+2;
							}
						}
					d+=4*(powsizex-letsizex);
					}

				glBindTexture(GL_TEXTURE_2D,gennum[255-g]);
				glTexImage2D(GL_TEXTURE_2D,0,GL_RGBA8,powsizex,powsizey,0,GL_RGBA,GL_UNSIGNED_BYTE,glyph);
				glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST);
				glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST);
				glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP);
				glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP);
				}

			//SPECIAL DISPENSATION FOR SPACE
			if(var_font)letwidth[' ']=letsizex>>1;
			else letwidth[' ']=letsizex;

			delete[] glyph;
			}
}

void texturest::create_texture_from_data(long data_pos)
{
	texture_data_pos=data_pos;
	texturefile.erase();

	if(enabler.texture_data.size()<=data_pos||data_pos==-1)return;

	texture_datast *td=enabler.texture_data[data_pos];

	if(td==NULL)return;

	width=td->width;
	height=td->height;
	alpha_texture=td->alpha_texture;

	glGenTextures(1,(GLuint*)&gennum);
	glBindTexture(GL_TEXTURE_2D,gennum);

	if(td->alpha_texture)
		{
		glTexImage2D(GL_TEXTURE_2D,0,GL_RGBA8,width,height,0,GL_RGBA,GL_UNSIGNED_BYTE,td->data);
		}
	else
		{
		glTexImage2D(GL_TEXTURE_2D,0,GL_RGBA8,width,height,0,GL_RGB,GL_UNSIGNED_BYTE,td->data);
		}

	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP);
}

void texturest::create_texture(const string &filename,char has_alpha_bit,char new_auto_alias)
{
	alpha_texture=has_alpha_bit;
	auto_alias=new_auto_alias;
	texturefile=filename;
	texture_data_pos=-1;

	bitmap_class bmp;

	if(alpha_texture)
		{
		bmp.data=enabler.load_bitmap_file_with_alpha(filename,&bmp.info);

		if(bmp.data!=NULL)
			{
			width=bmp.info.biWidth;
			height=bmp.info.biHeight;
			if(auto_alias)enabler.antialias(bmp.data,bmp.info.biWidth,bmp.info.biHeight);

			glGenTextures(1,(GLuint*)&gennum);
			glBindTexture(GL_TEXTURE_2D,gennum);
			glTexImage2D(GL_TEXTURE_2D,0,GL_RGBA8,bmp.info.biWidth,bmp.info.biHeight,0,GL_RGBA,GL_UNSIGNED_BYTE,bmp.data);
			glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST);
			glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST);
			glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP);
			glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP);
			}
		}
	else
		{
		bmp.data=enabler.load_bitmap_file(filename,&bmp.info);

		if(bmp.data!=NULL)
			{
			width=bmp.info.biWidth;
			height=bmp.info.biHeight;

			glGenTextures(1,(GLuint*)&gennum);
			glBindTexture(GL_TEXTURE_2D,gennum);
			glTexImage2D(GL_TEXTURE_2D,0,GL_RGBA8,bmp.info.biWidth,bmp.info.biHeight,0,GL_RGB,GL_UNSIGNED_BYTE,bmp.data);
			glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST);
			glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST);
			glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP);
			glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP);
			}
		}
}

void tilest::render()
{
	if(flag & TILEFLAG_LINE)
		{
		glDisable(GL_CULL_FACE);
		glDisable(GL_TEXTURE_2D);
		if(color_a!=1)
			{
			glEnable(GL_BLEND);
			glEnable(GL_ALPHA_TEST);
			glAlphaFunc(GL_NOTEQUAL,0);
			glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
			}
		else
			{
			glDisable(GL_BLEND);
			glDisable(GL_ALPHA_TEST);
			}

		glColor4f(color_r,color_g,color_b,color_a);

		glBegin(GL_LINES);
			glVertex2f(x,y);
			glVertex2f(force_dimx,force_dimy);
		glEnd();

		return;
		}

	if(flag & TILEFLAG_LINE_3D)
		{
		glDisable(GL_CULL_FACE);
		glDisable(GL_TEXTURE_2D);
		if(color_a!=1)
			{
			glEnable(GL_BLEND);
			glEnable(GL_ALPHA_TEST);
			glAlphaFunc(GL_NOTEQUAL,0);
			glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
			}
		else
			{
			glDisable(GL_BLEND);
			glDisable(GL_ALPHA_TEST);
			}

		glColor4f(color_r,color_g,color_b,color_a);

		glBegin(GL_LINES);
			glVertex3f(x,y,hor_stretch);
			glVertex3f(force_dimx,force_dimy,ver_stretch);
		glEnd();

		return;
		}

	if(flag & TILEFLAG_RECT)
		{
		glDisable(GL_CULL_FACE);
		glDisable(GL_TEXTURE_2D);
		if(color_a!=1)
			{
			glEnable(GL_BLEND);
			glEnable(GL_ALPHA_TEST);
			glAlphaFunc(GL_NOTEQUAL,0);
			glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
			}
		else
			{
			glDisable(GL_BLEND);
			glDisable(GL_ALPHA_TEST);
			}

		glColor4f(color_r,color_g,color_b,color_a);

		glBegin(GL_QUADS);
			glVertex2f(x,y);
			glVertex2f(x+force_dimx,y);
			glVertex2f(x+force_dimx,y+force_dimy);
			glVertex2f(x,y+force_dimy);
		glEnd();

		return;
		}

	long gennum;
	char alpha;
	double width,height;

	if(force_gennum>0)
		{
		gennum=force_gennum;
		alpha=force_alpha;
		width=force_dimx;
		height=force_dimy;
		}
	else
		{
		if(tex_pos<0||tex_pos>=enabler.texture.size())return;

		texturest *tx=enabler.texture[tex_pos];
		if(tx==NULL)return;
		if(tx->gennum==0)return;

		gennum=tx->gennum;
		alpha=tx->alpha_texture;
		width=tx->width;
		height=tx->height;
		}

	glDisable(GL_CULL_FACE);
	glEnable(GL_TEXTURE_2D);
	if(alpha||color_a!=1)
		{
		glEnable(GL_BLEND);
		glEnable(GL_ALPHA_TEST);
		glAlphaFunc(GL_NOTEQUAL,0);
		glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
		}
	else
		{
		glDisable(GL_BLEND);
		glDisable(GL_ALPHA_TEST);
		}

	glColor4f(color_r,color_g,color_b,color_a);
	glBindTexture(GL_TEXTURE_2D,gennum);

	float texminx=0;
	float texmaxx=1;
	float texminy=0;
	float texmaxy=1;
	if(flag & TILEFLAG_PIXRECT)
		{
		texminx=(float)pixmin_x/(float)width;
		texmaxx=(float)(pixmax_x+1)/(float)width;
		texminy=1-(float)(pixmax_y+1)/(float)height;
		texmaxy=1-(float)pixmin_y/(float)height;

		width=width*(pixmax_x-pixmin_x+1)/width;
		height=height*(pixmax_y-pixmin_y+1)/height;
		}

	if(flag & TILEFLAG_HORFLIP)
		{
		float swap=texminx;
		texminx=texmaxx;
		texmaxx=swap;
		}

	if(flag & TILEFLAG_VERFLIP)
		{
		float swap=texminy;
		texminy=texmaxy;
		texmaxy=swap;
		}

	if(hor_stretch!=1)width*=hor_stretch;
	if(ver_stretch!=1)height*=ver_stretch;

	glBegin(GL_QUADS);
		if(flag & TILEFLAG_ROTATE)
			{
			double cs,sn,cx,cy,ax,ay,nx,ny;

			cs=cos(rotate_angle);sn=sin(rotate_angle);
			cx=rotate_cenx+(double)width/2.0f;cy=rotate_ceny+(double)height/2.0f;

			glTexCoord2f(texminx,texmaxy);
			ax=x-cx;ay=y-cy;
			nx=cs*ax-sn*ay;ny=sn*ax+cs*ay;
			glVertex2f(cx+nx,cy+ny);

			glTexCoord2f(texmaxx,texmaxy);
			ax=x+(double)width-cx;ay=y-cy;
			nx=cs*ax-sn*ay;ny=sn*ax+cs*ay;
			glVertex2f(cx+nx,cy+ny);

			glTexCoord2f(texmaxx,texminy);
			ax=x+(double)width-cx;ay=y+(double)height-cy;
			nx=cs*ax-sn*ay;ny=sn*ax+cs*ay;
			glVertex2f(cx+nx,cy+ny);

			glTexCoord2f(texminx,texminy);
			ax=x-cx;ay=y+(double)height-cy;
			nx=cs*ax-sn*ay;ny=sn*ax+cs*ay;
			glVertex2f(cx+nx,cy+ny);
			}
		else
			{
			glTexCoord2f(texminx,texmaxy);
			glVertex2f(x,y);
			glTexCoord2f(texmaxx,texmaxy);
			glVertex2f(x+(double)width,y);
			glTexCoord2f(texmaxx,texminy);
			glVertex2f(x+(double)width,y+(double)height);
			glTexCoord2f(texminx,texminy);
			glVertex2f(x,y+(double)height);
			}
	glEnd();

	glDisable(GL_ALPHA_TEST);
}

void enablerst::print_string(const string &str,char centered,short length_lim,char crammed_lets)
{
	fontst *fnt=get_font(active_font_id);

	if(fnt==NULL)return;

	int s;
	long totalwidth=0;
	for(s=0;s<str.size();s++)
		{
		if(s>=length_lim)break;
		totalwidth+=fnt->letwidth[str[s]];
		if(s<(int)str.size()-1)
			{
			if(crammed_lets&&((str[s]>='A'&&str[s]<='Z')||(str[s]>='a'&&str[s]<='z'))&&
				((str[s+1]>='A'&&str[s+1]<='Z')||(str[s+1]>='a'&&str[s+1]<='z')))
				{
				totalwidth--;
				}
			}
		}

	//CENTER TEXT IF NECESSARY
	if(centered)
		{
		locx-=(totalwidth>>1);
		}

	//ADD TILES FOR EACH OF THE CHARS IN THE STRING
	for(s=0;s<str.size();s++)
		{
		if(s>=length_lim)break;
		add_gennum_tile(fnt->gennum[str[s]],fnt->powsizex,fnt->powsizey,fnt->letwidth[str[s]]-1,fnt->letsizey,1);
		locate(locx+fnt->letwidth[str[s]],locy);
		if(s<(int)str.size()-1)
			{
			if(crammed_lets&&((str[s]>='A'&&str[s]<='Z')||(str[s]>='a'&&str[s]<='z'))&&
				((str[s+1]>='A'&&str[s+1]<='Z')||(str[s+1]>='a'&&str[s+1]<='z')))
				{
				locate(locx-1,locy);
				}
			}
		}
}

void gridrectst::render()
{
	float apletsizex=dispx,apletsizey=dispy;
	float totalsizex=dispx*dimx;
	float totalsizey=dispy*dimy;
	float translatex=0,translatey=0;
	if(totalsizex>enabler.window_width||!black_space)apletsizex=(float)enabler.window_width/dimx;
	else translatex=(enabler.window_width-totalsizex)/2.0f;
	if(totalsizey>enabler.window_height||!black_space)apletsizey=(float)enabler.window_height/dimy;
	else translatey=(enabler.window_height-totalsizey)/2.0f;

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glTranslatef(translatex,translatey,0);

	if(trinum>0)
		{
		glEnable(GL_FOG);
		glFogf(GL_FOG_START,0);
		glFogf(GL_FOG_END,1000);
		glFogf(GL_FOG_MODE,GL_LINEAR);
		float fcolor[4]={0,0,0,0};
		glFogfv(GL_FOG_COLOR,fcolor);

		glMatrixMode(GL_PROJECTION);
		glPushMatrix();
		glLoadIdentity();
		gluPerspective(54.0f, enabler.window_width/enabler.window_height, 2.0f, 1000.0f);
		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		gluLookAt(115+160.0f*(float)sin(view_angle),-115-160.0f*(float)cos(view_angle),view_z,115,-115,150,0,0,1);

		glEnable(GL_DITHER);
		glShadeModel(GL_SMOOTH);
		glDisable(GL_TEXTURE_2D);
		glEnable(GL_BLEND);
		glEnable(GL_CULL_FACE);
		glEnable(GL_DEPTH_TEST);

		glBegin(GL_TRIANGLES);
			long t;
			for(t=0;t<trinum;t++)
				{
				glColor4fv(tricol[t]);
				glVertex3fv(tri[t]);
				}
		glEnd();

		glMatrixMode(GL_PROJECTION);
		glPopMatrix();
		glMatrixMode(GL_MODELVIEW);
		glPopMatrix();
		}

	glDisable(GL_DITHER);
	glShadeModel(GL_FLAT);
	glDisable(GL_BLEND);
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_CULL_FACE);
	glDisable(GL_DEPTH_TEST);

	//BACKGROUND COLORS
	long px,py;
	long d=0;
	for(px=0;px<dimx;px++)
		{
		for(py=0;py<dimy;py++,d++)
			{
			if(trinum>0)
				{
				if(py>=1&&py<=23&&px>=25&&px<=54)continue;
				}

			glColor3f(buffer_br[d],buffer_bg[d],buffer_bb[d]);

			glBegin(GL_QUADS);
				glVertex2f(px*apletsizex,py*apletsizey);
				glVertex2f((px+1)*apletsizex,py*apletsizey);
				glVertex2f((px+1)*apletsizex,(py+1)*apletsizey);
				glVertex2f(px*apletsizex,(py+1)*apletsizey);
			glEnd();
			}
		}

	//FOREGROUND
	glEnable(GL_TEXTURE_2D);
	glEnable(GL_ALPHA_TEST);
	glAlphaFunc(GL_NOTEQUAL,0);

	short tex_pos;
	texturest *txt;
	d=0;
	for(px=0;px<dimx;px++)
		{
		for(py=0;py<dimy;py++,d++)
			{
			if(trinum>0)
				{
				if(py>=1&&py<=23&&px>=25&&px<=54)continue;
				}

			glColor3f(buffer_r[d],buffer_g[d],buffer_b[d]);

			tex_pos=buffer_texpos[d];

			if(tex_pos<0||tex_pos>=enabler.texture.size())continue;
			txt=enabler.texture[tex_pos];
			if(txt==NULL)continue;
			if(txt->gennum==0)continue;

			glBindTexture(GL_TEXTURE_2D,txt->gennum);

			glBegin(GL_QUADS);
				glTexCoord2f(0,adjy);
				glVertex2f(px*apletsizex,py*apletsizey);
				glTexCoord2f(adjx,adjy);
				glVertex2f((px+1)*apletsizex,py*apletsizey);
				glTexCoord2f(adjx,0);
				glVertex2f((px+1)*apletsizex,(py+1)*apletsizey);
				glTexCoord2f(0,0);
				glVertex2f(px*apletsizex,(py+1)*apletsizey);
			glEnd();
			}
		}

	glDisable(GL_ALPHA_TEST);

	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
}

void cursesrectst::render(void)
{
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_CULL_FACE);
	glDisable(GL_DEPTH_TEST);

	//BACKGROUND COLORS
	float apletsizex=(float)enabler.window_width/dimx;
	float apletsizey=(float)enabler.window_height/dimy;

	int px,py;
	int d=0;
	for(px=0;px<dimx;px++)
		{
		for(py=0;py<dimy;py++,d++)
			{
			glColor3f(buffer_br[d],buffer_bg[d],buffer_bb[d]);

			glBegin(GL_QUADS);
				glVertex2f(px*apletsizex,py*apletsizey);
				glVertex2f((px+1)*apletsizex,py*apletsizey);
				glVertex2f((px+1)*apletsizex,(py+1)*apletsizey);
				glVertex2f(px*apletsizex,(py+1)*apletsizey);
			glEnd();
			}
		}

	//FOREGROUND
	glEnable(GL_TEXTURE_2D);
	glEnable(GL_ALPHA_TEST);
	glAlphaFunc(GL_NOTEQUAL,0);

	d=0;

	float tx=letsizex/powsizex;
	float ty=letsizey/powsizey;

	for(px=0;px<dimx;px++)
		{
		for(py=0;py<dimy;py++,d++)
			{
			if(gennum[buffer_char[d]]==0)continue;

			glColor3f(buffer_r[d],buffer_g[d],buffer_b[d]);
			glBindTexture(GL_TEXTURE_2D,gennum[buffer_char[d]]);

			glBegin(GL_QUADS);
				glTexCoord2f(0,ty);
				glVertex2f(px*apletsizex,py*apletsizey);
				glTexCoord2f(tx,ty);
				glVertex2f((px+1)*apletsizex,py*apletsizey);
				glTexCoord2f(tx,0);
				glVertex2f((px+1)*apletsizex,(py+1)*apletsizey);
				glTexCoord2f(0,0);
				glVertex2f(px*apletsizex,(py+1)*apletsizey);
			glEnd();
			}
		}

	glDisable(GL_ALPHA_TEST);
}

void cursesrectst::setcolor(short newf,short newb,char newbright)
{
	convert_to_rgb(r,g,b,newf,newbright);
	convert_to_rgb(br,bg,bb,newb,0);
}

void convert_to_rgb(float &r,float &g,float &b,char col,char bright)
{
	short ci=col+(bright!=0)*8;
	if(ci>=0&&ci<16)
		{
		r=enabler.ccolor[ci][0];
		g=enabler.ccolor[ci][1];
		b=enabler.ccolor[ci][2];
		}
}

void cursesrectst::usebuffer(char *newb,long buffersize)
{
	long localbufsize=dimx*dimy;

	int i,bi;
	for(i=0,bi=0;bi<buffersize&&i<localbufsize;i++,bi+=4)
		{
		buffer_char[i]=newb[bi];
		convert_to_rgb(buffer_r[i],buffer_g[i],buffer_b[i],newb[bi+1],newb[bi+3]);
		convert_to_rgb(buffer_br[i],buffer_bg[i],buffer_bb[i],newb[bi+2],0);
		}
}

cursesrectst::~cursesrectst()
{
	delete[] buffer_char;
	delete[] buffer_r;
	delete[] buffer_g;
	delete[] buffer_b;
	delete[] buffer_br;
	delete[] buffer_bg;
	delete[] buffer_bb;
}

cursesrectst *cursesrectst::create(long font_id,long x,long y,long dimx,long dimy)
{
	cursesrectst *newrect=new cursesrectst(font_id,x,y,dimx,dimy);
	return newrect;
}

gridrectst *gridrectst::create(long dimx,long dimy)
{
	gridrectst *newrect=new gridrectst(dimx,dimy);
	return newrect;
}

gridrectst::~gridrectst()
{
	delete[] buffer_texpos;
	delete[] buffer_r;
	delete[] buffer_g;
	delete[] buffer_b;
	delete[] buffer_br;
	delete[] buffer_bg;
	delete[] buffer_bb;
}

gridrectst::gridrectst(long newdimx,long newdimy)
{
	dimx=newdimx;
	dimy=newdimy;
	buffer_texpos=new long[dimx*dimy];
	buffer_r=new float[dimx*dimy];
	buffer_g=new float[dimx*dimy];
	buffer_b=new float[dimx*dimy];
	buffer_br=new float[dimx*dimy];
	buffer_bg=new float[dimx*dimy];
	buffer_bb=new float[dimx*dimy];
	trinum=0;
}

long enablerst::gridrect_create(long dimx,long dimy)
{
	gridrectst *rect=gridrectst::create(dimx,dimy);
		rect->id=next_gridrect_id;next_gridrect_id++;
	gridrect.push_back(rect);

	return rect->id;
}

fontst *fontst::create(const string &filename,long letx,long lety,char variable)
{
	fontst *newfont=new fontst(filename,letx,lety,variable);
	return newfont;
}

texturest *texturest::create(const string &filename,char has_alpha_bit,char new_auto_alias)
{
	texturest *newtex=new texturest(filename,has_alpha_bit,new_auto_alias);
	return newtex;
}

texturest *texturest::create_from_data(long data_pos,char has_alpha_bit)
{
	texturest *newtex=new texturest(data_pos,has_alpha_bit);
	return newtex;
}

texture_datast *texture_datast::create(short width,short height,char has_alpha_bit)
{
	texture_datast *newtexd=new texture_datast(width,height,has_alpha_bit);
	return newtexd;
}

void cursesrectst::setclipping(int x1,int x2,int y1,int y2)
{
	clipx[0]=x1;
	clipx[1]=x2;
	clipy[0]=y1;
	clipy[1]=y2;
}

fontst::fontst(const string &filename,long letx,long lety,char variable)
{
	memset(gennum,0,sizeof(unsigned int)*256);

	fontfile=filename;
	lx=letx;
	ly=lety;
	var_font=variable;
}

texturest::texturest(const string &filename,char has_alpha_bit,char new_auto_alias)
{
	alpha_texture=has_alpha_bit;
	auto_alias=new_auto_alias;
	texturefile=filename;
	texture_data_pos=-1;
	gennum=0;
}

texturest::texturest(long data_pos,char has_alpha_bit)
{
	alpha_texture=has_alpha_bit;
	auto_alias=0;
	texturefile.erase();
	texture_data_pos=data_pos;
	gennum=0;
}

void cursesrectst::adopt_textures(long newfont_id)
{
	font_id=newfont_id;
	fontst *font=enabler.get_font(font_id);
	if(font!=NULL)
		{
		memmove(gennum,font->gennum,sizeof(unsigned int)*256);

		letsizex=font->letsizex;
		letsizey=font->letsizey;
		powsizex=font->powsizex;
		powsizey=font->powsizey;
		}
	else memset(gennum,0,sizeof(unsigned int)*256);
}

cursesrectst::cursesrectst(long newfont_id,long newx,long newy,long newdimx,long newdimy)
{
	r=0;g=0;b=0;
	br=0;bg=0;bb=0;
	cursorx=0;cursory=0;
	setclipping(0,dimx-1,0,dimy-1);

	font_id=newfont_id;

	x=newx;
	y=newy;
	dimx=newdimx;
	dimy=newdimy;
	buffer_char=new unsigned char[dimx*dimy];
	buffer_r=new float[dimx*dimy];
	buffer_g=new float[dimx*dimy];
	buffer_b=new float[dimx*dimy];
	buffer_br=new float[dimx*dimy];
	buffer_bg=new float[dimx*dimy];
	buffer_bb=new float[dimx*dimy];
}

long enablerst::cursesrect_create(long font_id,long x,long y,long dimx,long dimy)
{
	cursesrectst *rect=cursesrectst::create(font_id,x,y,dimx,dimy);
		rect->id=next_cursesrect_id;next_cursesrect_id++;
	cursesrect.push_back(rect);

	return rect->id;
}

long enablerst::font_create(const string &filename,long letx,long lety,char variable)
{
	fontst *nfont=fontst::create(filename,letx,lety,variable);
		nfont->id=next_font_id;next_font_id++;
	font.push_back(nfont);

	return nfont->id;
}

long enablerst::texture_data_create(short newwidth,short newheight,char has_alpha_bit)
{
	texture_datast *ntexd=texture_datast::create(newwidth,newheight,has_alpha_bit);

	long slot;
	for(slot=(long)texture_data.size()-1;slot>=0;slot--)
		{
		if(texture_data[slot]==NULL)
			{
			texture_data[slot]=ntexd;
			return slot;
			}
		}

	texture_data.push_back(ntexd);
	return (long)texture_data.size()-1;
}

long enablerst::texture_create(const string &filename,char has_alpha_bit,char new_auto_alias)
{
	texturest *ntex=texturest::create(filename,has_alpha_bit,new_auto_alias);

	long slot;
	for(slot=(long)texture.size()-1;slot>=0;slot--)
		{
		if(texture[slot]==NULL)
			{
			texture[slot]=ntex;
			return slot;
			}
		}

	texture.push_back(ntex);
	return (long)texture.size()-1;
}

void enablerst::grayscale_texture_data(long pos)
{
	unsigned char *buff=get_texture_data(pos);
	if(buff==NULL)return;

	long bytes_per_pixel;
	if(texture_data[pos]->alpha_texture)bytes_per_pixel=4;
	else bytes_per_pixel=3;

	long dimx=texture_data[pos]->width;
	long dimy=texture_data[pos]->height;
	long ind=0,x,y;
	float norm=sqrt(3.0);
	float gx;
	long gr,gg,gb;
	for(x=0;x<dimx;x++)
		{
		for(y=0;y<dimy;y++)
			{
			gr=buff[ind];
			gg=buff[ind+1];
			gb=buff[ind+2];
			gx=(float)sqrt((float)(gr*gr+gg*gg+gb*gb))/norm;
			if(gx<0)gx=0;
			if(gx>255)gx=255;
			buff[ind]=(unsigned char)gx;

			ind+=bytes_per_pixel;
			}
		}
}

void enablerst::texture_create_multi_pdim(const string &filename,char has_alpha_bit,long *tex_pos,long *data_pos,long picx,long picy,
										  float &adjx,float &adjy,long &dispx,long &dispy)
{
	if(picx<=0||picy<=0)return;

	bitmap_class bmp;

	if(has_alpha_bit)bmp.data=enabler.load_bitmap_file_with_alpha(filename,&bmp.info);
	else bmp.data=enabler.load_bitmap_file(filename,&bmp.info);

	if(bmp.data!=NULL)
		{
		long sx,sy;
		long ndimx=bmp.info.biWidth/picx;
		long ndimy=bmp.info.biHeight/picy;
		dispx=ndimx;
		dispy=ndimy;
		long ind=0;
		long powsizex=ndimx;
		long powsizey=ndimy;
		int t=0;
		for(t=0;t<8;t++)
			{
			if(powsizex<=(1<<t)){powsizex=(1<<t);break;}
			}
		for(t=0;t<8;t++)
			{
			if(powsizey<=(1<<t)){powsizey=(1<<t);break;}
			}

		adjx=(float)ndimx/(float)powsizex;
		adjy=(float)ndimy/(float)powsizey;

		for(sy=0;sy<picy;sy++)
			{
			for(sx=0;sx<picx;sx++)
				{
				//MAKE DATA
				data_pos[ind]=texture_data_create(powsizex,powsizey,has_alpha_bit);
				long dp=data_pos[ind];
				unsigned char *buff=texture_data[dp]->data;

				long bitnum=3;
				if(has_alpha_bit)bitnum=4;

				long dind=(picy-sy-1)*ndimy*bmp.info.biWidth*bitnum+sx*ndimx*bitnum,dind2=0;
				long dx,dy;
				for(dy=ndimy-1;dy>=0;dy--)
					{
					for(dx=0;dx<ndimx;dx++)
						{
						buff[dind2]=bmp.data[dind];
						buff[dind2+1]=bmp.data[dind+1];
						buff[dind2+2]=bmp.data[dind+2];
						if(bitnum==4)buff[dind2+3]=bmp.data[dind+3];
						dind+=bitnum;
						dind2+=bitnum;
						}
					dind2+=bitnum*(powsizex-ndimx);
					dind+=bmp.info.biWidth*bitnum-ndimx*bitnum;
					}

				//MAKE TEXTURE
				tex_pos[ind]=texture_create_from_data(data_pos[ind],has_alpha_bit);
				ind++;
				}
			}
		}
}

void enablerst::texture_create_multi(const string &filename,char has_alpha_bit,long *tex_pos,long *data_pos,long ndimx,long ndimy)
{
	if(ndimx<=0||ndimy<=0)return;

	bitmap_class bmp;

	if(has_alpha_bit)bmp.data=enabler.load_bitmap_file_with_alpha(filename,&bmp.info);
	else bmp.data=enabler.load_bitmap_file(filename,&bmp.info);

	if(bmp.data!=NULL)
		{
		long sx,sy;
		long picx=bmp.info.biWidth/ndimx;
		long picy=bmp.info.biHeight/ndimy;
		long ind=0;
		for(sx=0;sx<picx;sx++)
			{
			for(sy=0;sy<picy;sy++)
				{
				//MAKE DATA
				data_pos[ind]=texture_data_create(ndimx,ndimy,has_alpha_bit);
				long dp=data_pos[ind];
				unsigned char *buff=texture_data[dp]->data;

				long bitnum=3;
				if(has_alpha_bit)bitnum=4;

				long dind=(picy-sy-1)*ndimy*bmp.info.biWidth*bitnum+sx*ndimx*bitnum,dind2=0;
				long dx,dy;
				for(dy=ndimy-1;dy>=0;dy--)
					{
					for(dx=0;dx<ndimx;dx++)
						{
						buff[dind2]=bmp.data[dind];
						buff[dind2+1]=bmp.data[dind+1];
						buff[dind2+2]=bmp.data[dind+2];
						if(bitnum==4)buff[dind2+3]=bmp.data[dind+3];
						dind+=bitnum;
						dind2+=bitnum;
						}
					dind+=bmp.info.biWidth*bitnum-ndimx*bitnum;
					}

				//MAKE TEXTURE
				tex_pos[ind]=texture_create_from_data(data_pos[ind],has_alpha_bit);
				ind++;
				}
			}
		}
}

long enablerst::texture_create_from_data(long data_pos,char has_alpha_bit)
{
	texturest *ntex=texturest::create_from_data(data_pos,has_alpha_bit);

	long slot;
	for(slot=(long)texture.size()-1;slot>=0;slot--)
		{
		if(texture[slot]==NULL)
			{
			texture[slot]=ntex;
			return slot;
			}
		}

	texture.push_back(ntex);
	return (long)texture.size()-1;
}

void cursesrectst::locate(int x,int y)
{
	cursorx=x;
	cursory=y;
}

void cursesrectst::addchar(char c)
{
	if(cursorx>=clipx[0]&&cursorx<=clipx[1]&&
		cursory>=clipy[0]&&cursory<=clipy[1])
		{
		int index=cursory+dimy*cursorx;
		buffer_char[index]=c;
		buffer_r[index]=r;
		buffer_g[index]=g;
		buffer_b[index]=b;
		buffer_br[index]=br;
		buffer_bg[index]=bg;
		buffer_bb[index]=bb;
		}
	cursorx++;
}

void cursesrectst::addstring(const string str)
{
	int s;
	for(s=0;s<str.length()&&cursorx<dimx;s++)
		{
		if(cursorx<0)
			{
			s-=cursorx;
			cursorx=0;
			if(s>=str.length())break;
			}

		addchar(str[s]);
		}
}

gridrectst *enablerst::get_gridrect(long rect_id)
{
	int r;
	for(r=0;r<gridrect.size();r++)
		{
		if(gridrect[r]->id==rect_id)return gridrect[r];
		}

	return NULL;
}

cursesrectst *enablerst::get_rect(long rect_id)
{
	int r;
	for(r=0;r<cursesrect.size();r++)
		{
		if(cursesrect[r]->id==rect_id)return cursesrect[r];
		}

	return NULL;
}

fontst *enablerst::get_font(long font_id)
{
	int r;
	for(r=0;r<font.size();r++)
		{
		if(font[r]->id==font_id)return font[r];
		}

	return NULL;
}

void enablerst::set_color(float r,float g,float b,float a)
{
	if(fade_t==0)
		{
		color_r=r;color_g=g;color_b=b;color_a=a;
		}
	else
		{
		color_r=r*(1.0f-fade_t)+fade_r*fade_t;
		color_g=g*(1.0f-fade_t)+fade_g*fade_t;
		color_b=b*(1.0f-fade_t)+fade_b*fade_t;
		color_a=a*(1.0f-fade_t)+fade_a*fade_t;
		}
}

void enablerst::cursesrect_set_font_id(long rect_id,long nfont_id)
{
	if(rect_id==0)return;

	cursesrectst *rect=get_rect(rect_id);

	if(rect!=NULL)
		{
		rect->font_id=nfont_id;
		}
}

void enablerst::cursesrect_setcolor(long rect_id,short f,short b,char bright)
{
	if(rect_id==0)return;

	cursesrectst *rect=get_rect(rect_id);

	if(rect!=NULL)
		{
		rect->setcolor(f,b,bright);
		}
}

void enablerst::cursesrect_usebuffer(long rect_id,char *b,long buffersize)
{
	if(rect_id==0)return;

	cursesrectst *rect=get_rect(rect_id);

	if(rect!=NULL)
		{
		rect->usebuffer(b,buffersize);
		}
}

void enablerst::locate(double fx,double fy,double fz)
{
	locx=fx;
	locy=fy;
	locz=fz;
}

void enablerst::add_line(double endx,double endy)
{
	tilest *newt;
	if(next_tile_slot==-1)
		{
		newt=new tilest;
		tile.push_back(newt);
		}
	else
		{
		newt=tile[next_tile_slot];

		next_tile_slot++;
		if(next_tile_slot>=tile.size())next_tile_slot=-1;
		}

	newt->flag=0;
	newt->x=locx;newt->y=locy;
	newt->force_dimx=endx;
	newt->force_dimy=endy;
	newt->flag|=TILEFLAG_LINE;
	newt->color_r=color_r;
	newt->color_g=color_g;
	newt->color_b=color_b;
	newt->color_a=color_a;
	if(buffer_draw)newt->flag|=TILEFLAG_BUFFER_DRAW;

	locx=endx;
	locy=endy;
}

void enablerst::add_line_3D(double endx,double endy,double endz)
{
	tilest *newt;
	if(next_tile_slot==-1)
		{
		newt=new tilest;
		tile.push_back(newt);
		}
	else
		{
		newt=tile[next_tile_slot];

		next_tile_slot++;
		if(next_tile_slot>=tile.size())next_tile_slot=-1;
		}

	newt->flag=0;
	newt->x=locx;newt->y=locy;newt->hor_stretch=locz;
	newt->force_dimx=endx;
	newt->force_dimy=endy;
	newt->ver_stretch=endz;
	newt->flag|=TILEFLAG_LINE_3D;
	newt->color_r=color_r;
	newt->color_g=color_g;
	newt->color_b=color_b;
	newt->color_a=color_a;
	if(buffer_draw)newt->flag|=TILEFLAG_BUFFER_DRAW;

	locx=endx;
	locy=endy;
	locz=endz;
}

void enablerst::add_rect(double endx,double endy)
{
	tilest *newt;
	if(next_tile_slot==-1)
		{
		newt=new tilest;
		tile.push_back(newt);
		}
	else
		{
		newt=tile[next_tile_slot];

		next_tile_slot++;
		if(next_tile_slot>=tile.size())next_tile_slot=-1;
		}

	newt->flag=0;
	newt->hor_stretch=1;
	newt->ver_stretch=1;
	newt->x=locx;newt->y=locy;
	newt->force_dimx=endx;
	newt->force_dimy=endy;
	newt->flag|=TILEFLAG_RECT;
	newt->color_r=color_r;
	newt->color_g=color_g;
	newt->color_b=color_b;
	newt->color_a=color_a;
	if(buffer_draw)newt->flag|=TILEFLAG_BUFFER_DRAW;
}

void enablerst::add_tile(long tex_pos,double rotate_cenx,double rotate_ceny,double rotate_angle,
						 long pixmin_x,long pixmax_x,long pixmin_y,long pixmax_y,
						 double hor_stretch,double ver_stretch)
{
	tilest *newt;
	if(next_tile_slot==-1)
		{
		newt=new tilest;
		tile.push_back(newt);
		}
	else
		{
		newt=tile[next_tile_slot];

		next_tile_slot++;
		if(next_tile_slot>=tile.size())next_tile_slot=-1;
		}

	newt->tex_pos=tex_pos;
	newt->x=locx;newt->y=locy;
	newt->color_r=color_r;
	newt->color_g=color_g;
	newt->color_b=color_b;
	newt->color_a=color_a;
	newt->flag=0;
	newt->force_gennum=-1;
	newt->force_dimx=0;
	newt->force_dimy=0;
	newt->force_alpha=0;
	if(pixmin_x!=-1)
		{
		newt->pixmin_x=pixmin_x;
		newt->pixmax_x=pixmax_x;
		newt->pixmin_y=pixmin_y;
		newt->pixmax_y=pixmax_y;
		newt->flag|=TILEFLAG_PIXRECT;
		}
	if(rotate_angle!=0)
		{
		newt->rotate_cenx=locx+rotate_cenx;
		newt->rotate_ceny=locy+rotate_ceny;
		newt->rotate_angle=rotate_angle;
		newt->flag|=TILEFLAG_ROTATE;
		}
	if(hor_stretch<0)
		{
		newt->flag|=TILEFLAG_HORFLIP;
		newt->hor_stretch=hor_stretch*-1;
		}
	else newt->hor_stretch=hor_stretch;
	if(ver_stretch<0)
		{
		newt->flag|=TILEFLAG_VERFLIP;
		newt->ver_stretch=ver_stretch*-1;
		}
	else newt->ver_stretch=ver_stretch;
	if(buffer_draw)newt->flag|=TILEFLAG_BUFFER_DRAW;
}

void enablerst::add_gennum_tile(long gennum,double dimx,double dimy,double letx,double lety,char alpha)
{
	tilest *newt;
	if(next_tile_slot==-1)
		{
		newt=new tilest;
		tile.push_back(newt);
		}
	else
		{
		newt=tile[next_tile_slot];

		next_tile_slot++;
		if(next_tile_slot>=tile.size())next_tile_slot=-1;
		}

	newt->tex_pos=-1;
	newt->x=locx;
	newt->y=locy+(dimy-lety);
	newt->flag=0;
	newt->color_r=color_r;
	newt->color_g=color_g;
	newt->color_b=color_b;
	newt->color_a=color_a;
	newt->force_gennum=gennum;
	newt->force_dimx=dimx;
	newt->force_dimy=dimy;

	newt->pixmin_x=0;
	newt->pixmax_x=letx;
	newt->pixmin_y=dimy-lety;
	newt->pixmax_y=dimy;
	newt->flag|=TILEFLAG_PIXRECT;

	newt->force_alpha=alpha;
	newt->hor_stretch=1;
	newt->ver_stretch=1;
	if(buffer_draw)newt->flag|=TILEFLAG_BUFFER_DRAW;
}

void enablerst::refresh_tiles()
{
	long t;
	for(t=(long)tile.size()-1;t>=0;t--)tile[t]->flag|=TILEFLAG_DEAD;
	if(tile.size()>0)next_tile_slot=0;
	else next_tile_slot=-1;
}

void enablerst::render_tiles()
{
	//SET UP THE VIEW
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	glTranslatef(-center_x,-center_y,0);

	//DRAW EVERYTHING
	int r;
	for(r=0;r<gridrect.size();r++)
		{
		gridrect[r]->render();
		}
	
	for(r=0;r<cursesrect.size();r++)
		{
		cursesrect[r]->render();
		}

	int t;
	for(t=0;t<tile.size();t++)
		{
		if(!(tile[t]->flag & TILEFLAG_BUFFER_DRAW)&&buffer_draw)continue;
		if((tile[t]->flag & TILEFLAG_BUFFER_DRAW)&&!buffer_draw)continue;

		if(tile[t]->flag & TILEFLAG_MODEL_TRANSLATE)
			{
			glTranslatef(tile[t]->color_r,tile[t]->color_g,tile[t]->color_b);
			continue;
			}

		if(tile[t]->flag & TILEFLAG_MODEL_PERSPECTIVE)
			{
			glMatrixMode(GL_PROJECTION);
			glLoadIdentity();
			gluPerspective(54.0f,(float)window_width/(float)window_height,2.0f,1000.0f);
			glMatrixMode(GL_MODELVIEW);
			glLoadIdentity();
			gluLookAt(tile[t]->color_r,tile[t]->color_g,tile[t]->color_b,
				tile[t]->x,tile[t]->y,tile[t]->hor_stretch,
				tile[t]->force_dimx,tile[t]->force_dimy,tile[t]->ver_stretch);
			continue;
			}

		if(tile[t]->flag & TILEFLAG_MODEL_ORTHO)
			{
			glMatrixMode(GL_PROJECTION);
			glLoadIdentity();
			glOrtho(0.0f,window_width,window_height,0,-1.0,1.0);
			glMatrixMode(GL_MODELVIEW);
			glLoadIdentity();
			continue;
			}

		if(!(tile[t]->flag & TILEFLAG_DEAD))
			{
			tile[t]->render();
			tile[t]->flag|=TILEFLAG_DEAD;
			}
		}

	//RESET THE NEXT SLOT
	if(!buffer_draw)
		{
		if(tile.size()>0)next_tile_slot=0;
		else next_tile_slot=-1;
		}
}

enabler_inputst enablerst::getinput(void)
{
	if(input[0].key!=0)
		{
		enabler_inputst ret=input[0];
		

		int i;
		for(i=1;i<100;i++)
			{
			input[i-1]=input[i];
			}
		input[99].key=0;

		return ret;
		}

	enabler_inputst ret;
		ret.key=0;
	return ret;
}

char get_slot_and_addbit_uchar(unsigned char &addbit,long &slot,long checkflag,long slotnum)
{
	if(checkflag<0)return 0;

	//FIND PROPER SLOT
	slot=checkflag>>3;
	if(slot>=slotnum)return 0;

	//FIND PROPER BIT IN THAT SLOT
	addbit=1<<(checkflag%8);

	return 1;
}

unsigned char *enablerst::get_texture_data(long pos)
{
	if(pos<0||pos>=texture_data.size())return NULL;

	if(texture_data[pos]==NULL)return NULL;

	return texture_data[pos]->data;
}

void enablerst::refresh_texture_data(long pos)
{
	if(pos<0||pos>=texture_data.size())return;

	int t;
	for(t=(int)texture.size()-1;t>=0;t--)
		{
		if(texture[t]==NULL)continue;

		if(texture[t]->texture_data_pos==pos)
			{
			texture[t]->remove_texture();
			texture[t]->create_texture_from_data(pos);
			}
		}
}

void enablerst::enable_fade(float r,float g,float b,float a,float t)
{
	fade_r=r;fade_g=g;fade_b=b;fade_a=a;fade_t=t;
}

void enablerst::disable_fade()
{
	fade_t=0;
}

void enablerst::copy_texture_data(long dest,long src,
								  long offx,long offy,
								  float rmult,float gmult,float bmult,
								  char use_trans,long *color_data,unsigned long flag)
{
	unsigned char *buff2=get_texture_data(src);
	if(buff2==NULL)return;

	copy_texture_data(dest,buff2,texture_data[src]->width,texture_data[src]->height,
			texture_data[src]->alpha_texture,offx,offy,rmult,gmult,bmult,use_trans,color_data,flag);
}

void enablerst::copy_texture_data(long dest,unsigned char *src,long srcx,long srcy,char srcalpha,
			long offx,long offy,
			float rmult,float gmult,float bmult,char use_trans,long *color_data,unsigned long flag)
{
	//GET BUFFERS
	unsigned char *buff1=get_texture_data(dest);
	unsigned char *buff2=src;

	if(buff1==NULL||buff2==NULL)return;

	//COMPUTE THE PARAMETERS
	long starty,endy,startx,endx,dstartx,dstarty;
	long dind,dind2,dincx,dincx2,dincy,dincy2;

		//START AND END
	startx=offx;endx=offx+srcx-1;
	starty=offy;endy=offy+srcy-1;

		//BOUND START AND END TO DESTINATION BUFFER
	dstartx=0;dstarty=0;
	if(startx<0){dstartx=startx*-1;startx=0;}
	if(starty<0){dstarty=starty*-1;starty=0;}
	if(endx>=texture_data[dest]->width)endx=texture_data[dest]->width-1;
	if(endy>=texture_data[dest]->height)endy=texture_data[dest]->height-1;

		//SET INCREMENTS BASED ON ALPHA BIT
	if(texture_data[dest]->alpha_texture)dincx=4;
	else dincx=3;
	if(srcalpha)dincx2=4;
	else dincx2=3;
	dincy=(texture_data[dest]->width+(endx-startx+1))*dincx;
	dincy2=(srcx+(endx-startx+1))*dincx2;

		//SET THE INITIAL POSITION IN THE BUFFER
	dind=startx*dincx+(texture_data[dest]->height-starty-1)*texture_data[dest]->width*dincx;
	dind2=dstartx*dincx2+(srcy-dstarty-1)*srcx*dincx2;

	//ERROR NOTE:
		//these flip commands can't handle the src over-running off the side of the dest buffer
	if(flag & COPYTEXTUREFLAG_VERFLIP)
		{
		dind2-=(endy-starty)*srcx*dincx2;
		if(!(flag & COPYTEXTUREFLAG_HORFLIP))dincy2=0;
		}

	if(flag & COPYTEXTUREFLAG_HORFLIP)
		{
		dind2+=(endx-startx)*dincx2;
		dincx2*=-1;
		if(!(flag & COPYTEXTUREFLAG_VERFLIP))dincy2=0;
		else dincy2*=-1;
		}

		//SET UP MULTIPLITERS
	long rmult_l=(long)(256.0f*rmult);//THE 256 IS NOT A PROBLEM SINCE IT IS A FLOAT ANYWAY
	long gmult_l=(long)(256.0f*gmult);
	long bmult_l=(long)(256.0f*bmult);

	//DO THE COPY
	long place_r,place_g,place_b,place_dist;
	long dx,dy;
	for(dy=starty;dy<=endy;dy++)
		{
		for(dx=startx;dx<=endx;dx++)
			{
			//RESPECT TRANSPARENCY
			if(use_trans)
				{
				if(srcalpha)
					{
					if(buff2[dind2+3]==0)
						{
						dind+=dincx;
						dind2+=dincx2;
						continue;
						}
					}
				}

			place_r=buff2[dind2];
			place_g=buff2[dind2+1];
			place_b=buff2[dind2+2];

			if(color_data!=NULL)
				{
				short slot=-1;

				if(place_r==place_g&&place_g==place_b&&place_r!=0){slot=COLOR_DATA_WHITE_R;place_dist=place_r;}
				if(place_r!=0&&place_g==0&&place_b==0){slot=COLOR_DATA_RED_R;place_dist=place_r;}
				if(place_g!=0&&place_b==0&&place_r==0){slot=COLOR_DATA_GREEN_R;place_dist=place_g;}
				if(place_b!=0&&place_r==0&&place_g==0){slot=COLOR_DATA_BLUE_R;place_dist=place_b;}
				if(place_r==place_g&&place_b==0&&place_r!=0){slot=COLOR_DATA_YELLOW_R;place_dist=place_r;}
				if(place_r==place_b&&place_g==0&&place_b!=0){slot=COLOR_DATA_MAGENTA_R;place_dist=place_r;}
				if(place_g==place_b&&place_r==0&&place_g!=0){slot=COLOR_DATA_CYAN_R;place_dist=place_g;}

				if(slot!=-1)
					{
					if(color_data[slot]!=-1)
						{
						place_r=(place_dist*color_data[slot])>>8;
						place_g=(place_dist*color_data[slot+1])>>8;
						place_b=(place_dist*color_data[slot+2])>>8;
						}
					}
				}

			//COPY IN DATA
			if(srcalpha)
				{
				if(use_trans&&buff2[dind2+3]!=255)
					{
					place_r=((place_r*(long)buff2[dind2+3]+(long)buff1[dind]*(long)(255-buff2[dind2+3]))>>8);
					place_g=((place_g*(long)buff2[dind2+3]+(long)buff1[dind+1]*(long)(255-buff2[dind2+3]))>>8);
					place_b=((place_b*(long)buff2[dind2+3]+(long)buff1[dind+2]*(long)(255-buff2[dind2+3]))>>8);
					}
				}
			if(texture_data[dest]->alpha_texture&&srcalpha)
				{
				if(!use_trans||buff2[dind2+3]==255)buff1[dind+3]=buff2[dind2+3];
				}

			if(rmult_l!=256)
				{
				buff1[dind]=(place_r*rmult_l)>>8;
				}
			else
				{
				buff1[dind]=place_r;
				}
			if(gmult_l!=256)
				{
				buff1[dind+1]=(place_g*gmult_l)>>8;
				}
			else
				{
				buff1[dind+1]=place_g;
				}
			if(bmult_l!=256)
				{
				buff1[dind+2]=(place_b*bmult_l)>>8;
				}
			else
				{
				buff1[dind+2]=place_b;
				}

			dind+=dincx;
			dind2+=dincx2;
			}
		dind-=dincy;
		dind2-=dincy2;
		}
}

void enablerst::remove_texture(long pos)
{
	if(pos>=0&&pos<texture.size())
		{
		if(texture[pos]==NULL)return;

		if(pos<650)
			{
			int a;
			a=0;
			}

		delete texture[pos];
		texture[pos]=NULL;
		}
}

void enablerst::remove_texture_data(long pos)
{
	if(pos>=0&&pos<texture_data.size())
		{
		if(texture_data[pos]==NULL)return;

		delete texture_data[pos];
		texture_data[pos]=NULL;
		}
}

void enablerst::read_pixels(int x,int y,int width,int height,unsigned char *buffer)
{
	glReadBuffer(GL_BACK);
	glReadPixels(x,y,width,height,GL_RGBA,GL_UNSIGNED_BYTE,buffer);
}

void enablerst::flip_texture_data(long pos,unsigned long flag)
{
	unsigned char *buff=get_texture_data(pos);
	if(buff==NULL)return;

	long bytes_per_pixel;
	if(texture_data[pos]->alpha_texture)bytes_per_pixel=4;
	else bytes_per_pixel=3;
	
	flip_uchar_array(buff,texture_data[pos]->width,texture_data[pos]->height,bytes_per_pixel,flag);
}

void enablerst::flip_uchar_array(unsigned char *buff,long dimx,long dimy,long bytes_per_pixel,unsigned long flag)
{
	unsigned char swap;
	long x,y,ind,incy,ind2,dimmult;

	if(flag & COPYTEXTUREFLAG_VERFLIP)
		{
		ind=0;
		incy=dimx*bytes_per_pixel;
		dimmult=(dimy-1)*incy;
		for(x=0;x<dimx;x++)
			{
			ind2=0;
			for(y=0;y<dimy>>1;y++)
				{
				swap=buff[ind+ind2];
				buff[ind+ind2]=buff[dimmult-ind2+ind];
				buff[dimmult-ind2+ind]=swap;

				if(bytes_per_pixel>1)
					{
					swap=buff[ind+1+ind2];
					buff[ind+1+ind2]=buff[dimmult-ind2+ind+1];
					buff[dimmult-ind2+ind+1]=swap;
					}
				if(bytes_per_pixel>2)
					{
					swap=buff[ind+2+ind2];
					buff[ind+2+ind2]=buff[dimmult-ind2+ind+2];
					buff[dimmult-ind2+ind+2]=swap;
					}
				if(bytes_per_pixel>3)
					{
					swap=buff[ind+3+ind2];
					buff[ind+3+ind2]=buff[dimmult-ind2+ind+3];
					buff[dimmult-ind2+ind+3]=swap;
					}
				ind2+=incy;
				}
			ind+=bytes_per_pixel;
			}
		}
	if(flag & COPYTEXTUREFLAG_HORFLIP)
		{
		ind=0;
		incy=dimx*bytes_per_pixel;
		dimmult=(dimx-1)*bytes_per_pixel;
		for(y=0;y<dimy;y++)
			{
			ind2=0;
			for(x=0;x<dimx>>1;x++)
				{
				swap=buff[ind+ind2];
				buff[ind+ind2]=buff[ind+dimmult-ind2];
				buff[ind+dimmult-ind2]=swap;

				if(bytes_per_pixel>1)
					{
					swap=buff[ind+ind2+1];
					buff[ind+ind2+1]=buff[ind+dimmult-ind2+1];
					buff[ind+dimmult-ind2+1]=swap;
					}
				if(bytes_per_pixel>2)
					{
					swap=buff[ind+ind2+2];
					buff[ind+ind2+2]=buff[ind+dimmult-ind2+2];
					buff[ind+dimmult-ind2+2]=swap;
					}
				if(bytes_per_pixel>3)
					{
					swap=buff[ind+ind2+3];
					buff[ind+ind2+3]=buff[ind+dimmult-ind2+3];
					buff[ind+dimmult-ind2+3]=swap;
					}
				ind2+=bytes_per_pixel;
				}
			ind+=incy;
			}
		}
}

void enablerst::add_perspective(float sx,float sy,float sz,
						float gx,float gy,float gz,
						float nx,float ny,float nz)
{
	tilest *newt;
	if(next_tile_slot==-1)
		{
		newt=new tilest;
		tile.push_back(newt);
		}
	else
		{
		newt=tile[next_tile_slot];

		next_tile_slot++;
		if(next_tile_slot>=tile.size())next_tile_slot=-1;
		}

	newt->color_r=sx;
	newt->color_g=sy;
	newt->color_b=sz;
	newt->x=gx;
	newt->y=gy;
	newt->hor_stretch=gz;
	newt->force_dimx=nx;
	newt->force_dimy=ny;
	newt->ver_stretch=nz;

	newt->flag=0;
	newt->flag|=TILEFLAG_MODEL_PERSPECTIVE;
}

void enablerst::add_translate(float sx,float sy,float sz)
{
	tilest *newt;
	if(next_tile_slot==-1)
		{
		newt=new tilest;
		tile.push_back(newt);
		}
	else
		{
		newt=tile[next_tile_slot];

		next_tile_slot++;
		if(next_tile_slot>=tile.size())next_tile_slot=-1;
		}

	newt->color_r=sx;
	newt->color_g=sy;
	newt->color_b=sz;

	newt->flag=0;
	newt->flag|=TILEFLAG_MODEL_TRANSLATE;
}

void enablerst::add_ortho()
{
	tilest *newt;
	if(next_tile_slot==-1)
		{
		newt=new tilest;
		tile.push_back(newt);
		}
	else
		{
		newt=tile[next_tile_slot];

		next_tile_slot++;
		if(next_tile_slot>=tile.size())next_tile_slot=-1;
		}

	newt->flag=0;
	newt->flag|=TILEFLAG_MODEL_ORTHO;
}

void text_system_file_infost::initialize_info()
{
	std::ifstream fseed(filename.c_str());
	if(fseed.is_open())
		{
		string str;

		while(std::getline(fseed,str))
			{
			if(str.length()>0)number++;
			}
		}
	else
		{
		string str;
		str="Error Initializing Text: ";
		str+=filename;
		errorlog_string(str);
		}
	fseed.close();
}

void text_system_file_infost::get_text(text_infost &text)
{
	text.clean();

	if(number==0)return;

	std::ifstream fseed(filename.c_str());
	if(fseed.is_open())
		{
		string str;

		int num=trandom(number);

		//SKIP AHEAD TO THE RIGHT SPOT
		while(num>0)
			{
			std::getline(fseed,str);
			num--;
			}

		//PROCESS THE STRING INTO TEXT ELEMENTS
		if(std::getline(fseed,str))
			{
			int curpos;
			string nextstr;
			char doing_long=0;

			text_info_elementst *newel;
			long end=str.length();
			
			while(end>0)
				{
				if(isspace(str[end-1]))end--;
				else break;
				}
			
			str.resize(end);

			for(curpos=0;curpos<end;curpos++)
				{
				//HANDLE TOKEN OR ENDING
					//TWO FILE TOKENS IN A ROW MEANS LONG
					//ONE MEANS STRING
				if(str[curpos]==file_token || curpos==end-1)
					{
					if(str[curpos]!=file_token)nextstr+=str[curpos];

					//HAVE SOMETHING == SAVE IT
					if(!nextstr.empty())
						{
						if(doing_long)
							{
							newel=new text_info_element_longst(atoi(nextstr.c_str()));
							text.element.push_back(newel);
							doing_long=0;
							}
						else
							{
							newel=new text_info_element_stringst(nextstr);
							text.element.push_back(newel);
							}

						nextstr.erase();
						}
					//STARTING A LONG
					else
						{
						doing_long=1;
						}
					}
				//JUST ADD IN ANYTHING ELSE
				else
					{
					nextstr+=str[curpos];
					}
				}
			}
		}
	fseed.close();
}

void text_boxst::add_paragraph(const string &src,long para_width)
{
	stringvectst sp;
	sp.add_string(src);
	add_paragraph(sp,para_width);
}

void text_boxst::add_paragraph(stringvectst &src,long para_width)
{
	//USE THE WIDTH OF THE CURRENT FONT
	fontst *fnt=enabler.get_font(enabler.active_font_id);

	if(fnt==NULL)return;

	//ADD EACH OF THE STRINGS ON IN TURN
	string curstr;
	//string curword;
	long strlength=0;
	//long wordlength=0;
	int s,pos;
	for(s=0;s<src.str.size();s++)
		{
		//GRAB EACH WORD, AND SEE IF IT FITS, IF NOT START A NEW LINE
		for(pos=0;pos<src.str[s]->dat.size();pos++)
			{
			//ADD TO WORD
			curstr+=src.str[s]->dat[pos];
			strlength+=fnt->letwidth[src.str[s]->dat[pos]];

			//IF TOO LONG, CUT BACK TO FIRST SPACE
			if(strlength>para_width)
				{
				long opos=pos;

				int minus=0;
				do
					{
					pos--;
					minus++;
					}while(src.str[s]->dat[pos]!=' '&&pos>0);

				//IF WENT ALL THE WAY BACK, INTRODUCE A SPACE
				if(minus==curstr.size())
					{
					src.str[s]->dat.insert(opos-1," ");
					}
				else
					{
					curstr.resize(curstr.size()-minus);

					text.add_string(curstr);
					}
				curstr.erase();
				strlength=0;
				}
			}
		}

	//FLUSH FINAL BIT
	if(!curstr.empty())text.add_string(curstr);
}

void curses_text_boxst::add_paragraph(const string &src,long para_width)
{
	stringvectst sp;
	sp.add_string(src);
	add_paragraph(sp,para_width);
}

void curses_text_boxst::add_paragraph(stringvectst &src,long para_width)
{
	//ADD EACH OF THE STRINGS ON IN TURN
	string curstr;
	int s,pos;
	for(s=0;s<src.str.size();s++)
		{
		//GRAB EACH WORD, AND SEE IF IT FITS, IF NOT START A NEW LINE
		for(pos=0;pos<src.str[s]->dat.size();pos++)
			{
			//ADD TO WORD
			curstr+=src.str[s]->dat[pos];

			//IF TOO LONG, CUT BACK TO FIRST SPACE
			if(curstr.length()>para_width)
				{
				long opos=pos;

				int minus=0;
				do
					{
					pos--;
					minus++;
					}while(src.str[s]->dat[pos]!=' '&&pos>0);

				//IF WENT ALL THE WAY BACK, INTRODUCE A SPACE
				if(minus==curstr.size())
					{
					src.str[s]->dat.insert(opos-1," ");
					}
				else
					{
					curstr.resize(curstr.size()-minus);

					text.add_string(curstr);
					}
				curstr.erase();
				}
			}
		}

	//FLUSH FINAL BIT
	if(!curstr.empty())text.add_string(curstr);
}

void enablerst::antialias(unsigned char *dat,long srcx,long srcy,char border)
{
	long pos=3,x,y;
	for(y=0;y<srcy;y++)
		{
		for(x=0;x<srcx;x++,pos+=4)
			{
			if(dat[pos]!=0)
				{
				if(x>0&&y>0&&x<srcx-1&&y<srcy-1)
					{
					if(dat[pos-4]==0||
						dat[pos+4]==0||
						dat[pos-4*srcx]==0||
						dat[pos+4*srcx]==0)
						{
						dat[pos]=128;
						}
					}
				else if(border)dat[pos]=128;
				}
			}
		}
}

char enablerst::register_window_class (Application* application)
{
	// Here we'll load up the nib.
	if (CreateNibReference(CFSTR("main"), &application->mainNib) != 0)
		return false;
		
	// Set the menubar
	if (SetMenuBarFromNib(application->mainNib, CFSTR("MenuBar")) != 0)
		return false;
	
	return true;
}

void SetCurDirToAppFolder()
{
    CFURLRef bundleUrl = CFBundleCopyBundleURL(CFBundleGetMainBundle());
	char appPath[1024] = {0};
	if (CFURLGetFileSystemRepresentation(bundleUrl, true, (UInt8*)appPath, 1023))
	{
		char* p = strrchr(appPath, '/');	// chop off the app name
		*p = NULL;
		chdir(appPath);
		CFRelease(bundleUrl);
	}
	else
	{
		MessageBox(HWND_DESKTOP, "Error setting current dir!", "Error", MB_OK | MB_ICONEXCLAMATION);
		CFRelease(bundleUrl);
		exit(1);
	}
}

void CreateRenderContexts()
{
	GDHandle gdhDisplay;
	DMGetGDeviceByDisplayID((DisplayIDType)kCGDirectMainDisplay, &gdhDisplay, true);

	AGLPixelFormat pfWindowed = aglChoosePixelFormat(&gdhDisplay, 1, attrsWindowed);
	if(!pfWindowed) goto gfx_fail;
	
	contextWindowed = aglCreateContext(pfWindowed, NULL);
	if (!contextWindowed) goto gfx_fail;

   aglDestroyPixelFormat(pfWindowed);
	return;
	
gfx_fail:
	MessageBox(HWND_DESKTOP, "Error configuring graphics!", "Error", MB_OK | MB_ICONEXCLAMATION);
	if(contextWindowed) aglDestroyContext(contextWindowed);
	if(pfWindowed) aglDestroyPixelFormat(pfWindowed);
	exit(1);
}

void DestroyRenderContexts()
{
	if(contextFullscreen) aglDestroyContext(contextFullscreen);
	if(contextWindowed) aglDestroyContext(contextWindowed);
}

int main (int argc, char* argv[])
{
	SetCurDirToAppFolder();
	CreateRenderContexts();
	
	string cmdLine;
	for (int i = 1; i < argc; ++i)
	{
      // Apple adds a command-line argument -psn_####, we want to
      // make sure it does not get appended to enabler.command_line
      if (strncmp(argv[i], "-psn", 4) == 0)
         continue;
      
		cmdLine += argv[i];
		cmdLine += " ";
	}
	enabler.command_line = cmdLine;
	
	for (int i = 0; i < 255; ++i)
	{
		sKeyState[i] = (kKeyIsUp | kKeyWasDown);
	}
	
	int result = enabler.loop(NULL);
	
	DestroyRenderContexts();
	return result;
}

// -------------------------

DWORD GetTickCount()	// returns ms since system startup
{
	UnsignedWide usec;
	Microseconds(&usec);
	
	unsigned long long ms = usec.hi;
	ms <<= 32;
	ms += usec.lo;
	ms /= 1000;
	
	return (DWORD) ms;
}

BOOL CreateDirectory(const char* pathname, void*)
{
	if (mkdir(pathname, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH))
	{
      if (errno != EEXIST)
         fprintf(stderr, "mkdir(\"%s\") failed, errno %d\n", pathname, errno);
		return false;
	}
	else
		return true;
}

BOOL DeleteFile(const char* filename)
{
	return !unlink(filename);
}

void ZeroMemory(void* dest, int len)
{
	memset(dest, 0, len);
}

BOOL QueryPerformanceCounter(LARGE_INTEGER* performanceCount)
{
	UnsignedWide usec;
	Microseconds(&usec);
	
	performanceCount->LowPart = usec.lo;
	performanceCount->HighPart = usec.hi;
	return true;
}

BOOL QueryPerformanceFrequency(LARGE_INTEGER* performanceCount)
{
	// As we are using Microseconds, the freq is fixed: 10^6 per sec
	performanceCount->QuadPart = 1000 * 1000;
	return true;
}

SHORT GetKeyState(int virtKey)
{
/*	// MSB: 1 if down
	// LSB: 1 if toggled on (e.g. CAPS Lock)
	SHORT res = 0;

	if ((sKeyState[virtKey] & kKeyIsUp) == 0)
		res |= 0x8000u;

	return res;
*/
	// Uuhh... okay, this seems to work.
	return ((sKeyState[virtKey] & kKeyIsUp) == 0) ? -1 : 0;
}

int ShowCursor(BOOL show)
{
	static int displayCount = 0;

	if (show)
	{
		displayCount += 1;
		if (displayCount == 0)
			CGDisplayShowCursor(kCGDirectMainDisplay);
	}
	else
	{
		if (displayCount == 0)
			CGDisplayHideCursor(kCGDirectMainDisplay);
		displayCount -= 1;
	}
	
	return displayCount;
}

char* itoa(int value, char* result, int base)
{
	// check that the base is valid
	if (base < 2 || base > 16) { *result = NULL; return result; }
	
	char* out = result;
	int quot = value;
	
	do
	{
		*out = "0123456789abcdef"[ std::abs(quot % base) ];
		++out;
		quot /= base;
	}
	while (quot);
	
	if (value < 0) *out++ = '-';
	
	std::reverse(result, out);
	*out = NULL;
	return result;
}

int MessageBox(HWND hwnd, const char* text, const char* caption, UINT type)
{
	AlertType alert = kAlertNoteAlert;
	if (type & MB_ICONEXCLAMATION)alert = kAlertStopAlert;
	
	CFStringRef strText = CFStringCreateWithCStringNoCopy(NULL, text, kCFStringEncodingWindowsLatin1, kCFAllocatorNull);
	CFStringRef strCaption = CFStringCreateWithCStringNoCopy(NULL, caption, kCFStringEncodingWindowsLatin1, kCFAllocatorNull);
	
	AlertStdCFStringAlertParamRec params;
	params.version = kStdCFStringAlertVersionOne;
	params.movable = true;
	params.helpButton = false;
	params.defaultText = (type & MB_YESNO) ? CFSTR("Yes") : CFSTR("OK");
	params.cancelText = (type & MB_YESNO) ? CFSTR("No") : NULL;
	params.otherText = NULL;
	params.defaultButton = kAlertStdAlertOKButton;
	params.cancelButton = (type & MB_YESNO) ? kAlertStdAlertCancelButton : NULL;
	params.position = kWindowCenterMainScreen;
	params.flags = 0;
	
	DialogRef dlg;
	OSStatus err = CreateStandardAlert(alert, strCaption, strText, &params, &dlg);
	require_noerr(err, MessageBoxFailed);
   
	DialogItemIndex itemHit;
	err = RunStandardAlert(dlg, NULL, &itemHit);
	require_noerr(err, MessageBoxFailed);
	
	CFRelease(strCaption);
	CFRelease(strText);
	
	if (type & MB_YESNO)
		return (itemHit == kAlertStdAlertOKButton) ? IDYES : IDNO;
	else
		return IDOK;

MessageBoxFailed:
	CFRelease(strCaption);
	CFRelease(strText);
	return 0;
}
